/******************************************************************************/
/*									      */
/*	ctk_command_line.cpp	    				      	      */
/*									      */
/*	Command Line Interpretation				      	      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			              */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include "ctk_command_line.hh"
 
#include <iosfwd>

#include "ctk_local.hh"

#include "ctk_function_classes.hh"
#include "ctk_error.hh"
#include "ctk_block.hh"
#include "ctk_script.hh"
#include "ctk_CTK_reader.hh"
 

string expand_environment_variables(const string &string_in) {

  string string_out, varname;
  Character ch=0;
  UInteger i=0;
  Boolean in_var_name=false;

  while (i<string_in.size()) {
    ch=string_in[i];

    if (in_var_name) {
      if (isalnum(ch) || ch=='_') {
	varname+=ch;
      } else {
	in_var_name=false;
	char *value=getenv(varname.c_str());
	if (value==0) { 
	  string_out+='$'; string_out+=varname;
        } else {
	  string_out+=string(value);
	}
	if (ch=='$')
	  in_var_name=true;
      }
    }
    if (ch=='$') {
      in_var_name=true;
      varname.resize(0);
    } else if (!in_var_name) {
      string_out+=ch;
    }
    ++i;
  }
  
  if (in_var_name) {
    char  *value=getenv(varname.c_str());
    if (value==0) { 
      string_out+='$'; string_out+=varname;
    } else {
      string_out+=string(value);
    }
  }
  
  return string_out;
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKCommandOption  	      				      */
/*									      */
/******************************************************************************/

CTKCommandOption::CTKCommandOption(const string &aname) {
  name=aname;
  occurrences=0;
  current_occurrence=1;
  nargs=0;
  option_list.push_back(new StringVector());
};

void CTKCommandOption::add_argument(const string &this_argument_value) {
  option_list[0]->push_back(this_argument_value);
  ++nargs;
}

Integer CTKCommandOption::set_arguments(Integer argc, const char *argv[]) {
  
  if (argc<nargs) {
    cerr << "option \"-" << name << "\" requires " << nargs << " arguments" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  StringVector *argument_value=option_list[occurrences];
  for (Integer i=0; i<nargs; ++i) {
    (*argument_value)[i]=string(expand_environment_variables(argv[i]));
  }
  
  ++occurrences;
  option_list.push_back(new StringVector(nargs));

  return argument_value->size();
}


string CTKCommandOption::get_argument(Integer narg) {
  if (narg>nargs) {
    cerr << "option named \"-" << name << "\" only has " << nargs << " arguments" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  return (*option_list[current_occurrence-1])[narg-1];
}


Boolean CTKCommandOption::is_present() {
  return occurrences>0;
}

void CTKCommandOption::reset_to_first_occurrence() {
  current_occurrence=1;
};

Boolean CTKCommandOption::next_occurrence() {
  if (current_occurrence>=occurrences)
    return false;
  ++current_occurrence;
  return true;
};

/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKCommandLine    	      				      */
/*									      */
/******************************************************************************/

CTKCommandLine::CTKCommandLine(const string &format, int argc, const char *argv[]) {

  read_format(format); 

  parse(argc, argv);

  argument_list.resize(0);

  // Add the name of the script as the first entry on the argument list
  vector<string> alist;
  alist.push_back(argv[1]); 
  argument_list.push_back(alist);
  
  // Run through arguments, for -S add the list, for -??? skip, for others just add as they are

  for (UInteger j=1; j<argument.size(); ++j) {
    string arg=argument[j];
    if (arg=="-S") {
      // Add argument list
      CTKScriptElement filelist = get_option_argument("S");
      next_occurrence("S");
      add_to_argument_list(filelist.get_string());
    } else if (is_option(arg)) { 
      // Do nothing
    } else {
      // Add command line argument
      for (UInteger i=0; i<argument_list.size(); ++i) {
	argument_list[i].push_back(arg);
      }
    }
  }

  // Set position to first line in the argument list
  current_argument_list=0;
}


void CTKCommandLine::add_to_argument_list(const string &filelist) {

  std::ifstream ifstr(filelist.c_str());

  if (ifstr==NULL) {
    cerr << "Cannot open parameter list file: " << filelist << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  Character buffer[CTK_CHAR_BUFFER_SIZE];
  
  vector<string> tmplist;
  while (ifstr.getline(buffer,CTK_CHAR_BUFFER_SIZE,'\n')) {
    string line(buffer);
    if (line.size()>0 && line[0]!='#') {
      string exline=expand_environment_variables(line);
      tmplist.push_back(exline);
    }
  }

  unsigned int list_size = argument_list.size();

  if (list_size<=1) {
    // First -S parameter to be processed
    for (UInteger i=list_size; i<tmplist.size(); ++i)
      argument_list.push_back(argument_list[0]);
  } else if (list_size!=tmplist.size()) {
    // Previous -S parameter pointed to a file with a different number of lines
    cerr << "Argument lists have different sizes" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  // Apend the current -S to the exist argument lists
  for (unsigned int i=0; i<tmplist.size(); ++i)
    argument_list[i].push_back(tmplist[i]);
 

}

Boolean CTKCommandLine::is_option(const string &word) {
// Return's true if the string looks like an option identifier e.g. -help or -param etc.
// Note .. can't just check for '-' as -ve numbers (e.g. -20 or -.04 etc) are parameter values and not options
  return (word.size()>1 && word[0]=='-' && word[1]!='.' && !isdigit(word[1]));
}

Boolean CTKCommandLine::parse(int argc, const char *argv[]) {

  Integer i=1;

  while (i<argc) {
    string word=argv[i];
    // Look for options (i.e. starting with '-') but not to be confused with -ve arguments (e.g -.01)
    if (is_option(word)) {
      CTKCommandOption* option=option_map[string(word,1)];
      if (option==NULL) {
	cerr << "CTKCommandLine::parse:  unknown option - " << word << endl;
	throw(CTKError(__FILE__, __LINE__));
      }
      argument.push_back(word); 
      ++i;
      i+=option->set_arguments(argc-i, argv+i);
      
    } else {
      argument.push_back(expand_environment_variables(word));
      ++i;
    }
  }

  return true;

}




UInteger CTKCommandLine::argument_count() {
  return argument_list[current_argument_list].size();
}

string CTKCommandLine::get_argument(UInteger n) {
  if (n>argument_count()) {
    cerr << "Command line only has " << argument_count() << " arguments!" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
    
  return argument_list[current_argument_list][n-1];
}

vector<string> CTKCommandLine::get_arguments() {
  return argument_list[current_argument_list];
}

Boolean CTKCommandLine::next_argument_list() {
  if (current_argument_list==argument_list.size()-1) return false;

  ++current_argument_list;

  return true;
}

void CTKCommandLine::reset_to_first_argument_list() {
  current_argument_list=0;
}


Boolean CTKCommandLine::option_is_valid(const string &option_name) {
  CTKCommandOption *option=option_map[option_name];
  return (option!=NULL);
}

Boolean CTKCommandLine::option_is_present(const string &option_name) {
  CTKCommandOption *option=option_map[option_name];
  if (option==NULL) {
    cerr << "Option \"-" << option_name << " not valid " << endl;
  }

  return option->is_present();
}

string CTKCommandLine::get_option_argument(const string &option_name, Integer noption_arg/*=1*/) {
  CTKCommandOption *option=option_map[option_name];
  if (option==NULL) {
    cerr << "Option \"-" << option_name << " not valid " << endl;
  }

  return option->get_argument(noption_arg);
}

void CTKCommandLine::reset_to_first_occurrence(const string &option_name) {
  CTKCommandOption *option=option_map[option_name];
  if (option==NULL) {
    cerr << "Option \"-" << option_name << " not valid " << endl;
  }

  option->reset_to_first_occurrence();
}

Boolean CTKCommandLine::next_occurrence(const string &option_name) {
  CTKCommandOption *option=option_map[option_name];
  if (option==NULL) {
    cerr << "Option \"-" << option_name << " not valid " << endl;
  }

  return option->next_occurrence();

}

//
// Private methods
//

Boolean CTKCommandLine::read_format(const string &format) {
  vector<string*> word_list = make_word_list(format);

  CTKCommandOption *this_option=NULL;
  
  for (vector<string*>::iterator wordp=word_list.begin(); wordp!=word_list.end(); ++wordp) {
    string &word=(**wordp);
    if (word[0]=='-') {
      if (word.size()<2) {
	cerr << "CTKCommandLine::read_format - invalid format string: " << format << endl;
	throw(CTKError(__FILE__, __LINE__));
      }
      this_option=new CTKCommandOption(string(word,1));
      option_map[string(word,1)]=this_option;
    } else {
      if (this_option==NULL) {
	cerr << "CTKCommandLine::read_format - invalid format string: " << format << endl;
	throw(CTKError(__FILE__, __LINE__));
      }
      this_option->add_argument(word);
    }
  }

  sequence_delete(word_list.begin(), word_list.end());
  
  return false;
}


vector<string*> CTKCommandLine::make_word_list(const string &words) {

  Character ch=0;
  vector<string*> word_list;
  
  string::const_iterator chp=words.begin();
  while (chp!=words.end()) {
    while (chp!=words.end() && isspace(*chp)) ++chp;
    if (chp!=words.end()) {
      string *word=new string();
      while ((chp!=words.end())&&(isspace(ch=*chp)==false)) {
        (*word)+=ch;
        ++chp;
      }      
      word_list.push_back(word);
    }
  }

  return word_list;
 
}

//----------------------- Non-class methods 

void setParametersFromCommandline(Block &block, CTKCommandLine &command_line, CTKReader &script) {

  vector<string> command_line_arguments;
  
  command_line_arguments.push_back("\0");
  for (UInteger i=2; i<=command_line.argument_count(); ++i) {
    command_line_arguments.push_back(command_line.get_argument(i));
  }

  vector<UnresolvedParam> unresolved_params=script.get_unresolved_parameters();

  for (UInteger i=0; i<unresolved_params.size(); ++i) {
    string value;
    int fixed_component_index=0;
    UnresolvedParam unresolved_param = unresolved_params[i];
    for (UInteger j=0; j<unresolved_param.components.size(); ++j) {
      int component_number = unresolved_param.components[j];
      if (component_number==0) {
	// concatentate next fixed string segment
	value=value+unresolved_param.fixed_components[fixed_component_index++];
      } else {
	// concatentate command line param value
	value=value+command_line_arguments[component_number];
      }
    }
    //    cerr << "setting " << unresolved_param.name << " to " <<  value.c_str() << "\n";
    block.set_parameter(unresolved_param.name, value.c_str());
  }
  
}

/* End of ctk_command_line.cpp */
