/******************************************************************************/
/*									      */
/*	ctk_data_descriptor.cpp	    					      */
/*									      */
/*	Used by sockets to describe the data they are passing		      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <algorithm>
#include <numeric>

#include "ctk_local.hh"

#include "ctk_function_classes.hh"
#include "ctk_data_descriptor.hh"

/******************************************************************************/
/*									      */
/*	CLASS NAME: DataDescriptor		      			      */
/*									      */
/******************************************************************************/


DataDescriptor::DataDescriptor(){
  sample_data_flag=true;
  n_dimensions=0;
  dimension_sizes.resize(0);
  strides.resize(0);
}

DataDescriptor::DataDescriptor(const DataDescriptor &a_descriptor) {

  dimension_sizes=a_descriptor.dimension_sizes;
  strides=a_descriptor.strides;
  n_dimensions=a_descriptor.n_dimensions;
  sample_data_flag=a_descriptor.sample_data_flag;
  
  vector<DimensionDescriptor *>::const_iterator ddit;
  
  for (ddit=a_descriptor.dimension.begin(); ddit!=a_descriptor.dimension.end(); ++ddit) {
    dimension.push_back(new DimensionDescriptor(**ddit));
  }
}

DataDescriptor::~DataDescriptor(){
  sequence_delete(dimension.begin(), dimension.end());
};


void DataDescriptor::add_inner_dimension(const string &name, const CTKVector &axis) {
  //  sample_data_flag=false;
  if (axis.size()>1)
    sample_data_flag=false;
  ++n_dimensions;
  dimension_sizes.push_back(axis.size());
  dimension.push_back(new DimensionDescriptor(name, axis));
}

void DataDescriptor::add_outer_dimension(const string &name, const CTKVector &axis) {
  //  sample_data_flag=false;

  if (axis.size()>1)
    sample_data_flag=false;
  ++n_dimensions;
  
  dimension_sizes.insert(dimension_sizes.begin(), axis.size());
  dimension.insert(dimension.begin(), new DimensionDescriptor(name, axis)); 
}

Boolean DataDescriptor::add_dimension_at(Integer n, const string &name, const CTKVector &axis) {
  //  sample_data_flag=false;
  if (axis.size()>1)
    sample_data_flag=false;
  ++n_dimensions;
  dimension_sizes.insert(dimension_sizes.begin()+n-1, axis.size());
  dimension.insert(dimension.begin()+n-1, new DimensionDescriptor(name, axis));
  return true;
}

Boolean DataDescriptor::replace_dimension(const string &input_name, const string &output_name, const CTKVector &axis) {
  const DimensionDescriptor *dd=get_dimension(input_name);
  if (dd==NULL) return false;
  return replace_dimension_at(dd->get_index(), output_name, axis);
}

Boolean DataDescriptor::replace_dimension_at(Integer n, const string &name, const CTKVector &axis) {
  if (remove_dimension(n)==false) return false;
  return add_dimension_at(n, name, axis);
}

Boolean DataDescriptor::remove_inner_dimension() {
  if (n_dimensions>0) {
    --n_dimensions;
    //    if (n_dimensions==0) sample_data_flag=true;
    delete dimension.back();
    dimension.pop_back();
    dimension_sizes.pop_back();
    if (get_storage()==1) sample_data_flag=true;
    return true;
  } else return false;
}

Boolean DataDescriptor::remove_outer_dimension() {
  if (n_dimensions>0) {
    --n_dimensions;
    //    if (n_dimensions==0) sample_data_flag=true;
    delete dimension.front();
    dimension.erase(dimension.begin());
    dimension_sizes.erase(dimension_sizes.begin());
    if (get_storage()==1) sample_data_flag=true;
    return true;
  } else return false;
}

Boolean DataDescriptor::remove_dimension(const string &axis_name) {
  const DimensionDescriptor *dd=get_dimension(axis_name);
  if (dd==NULL) return false;
  return remove_dimension(dd->get_index());
}

Boolean DataDescriptor::remove_dimension(Integer n) {
  if (n<1||n>n_dimensions) return false;
  --n_dimensions;
  //  if (n_dimensions==0) sample_data_flag=true;
  vector<DimensionDescriptor *>::iterator ddit=dimension.begin()+n-1;
  delete *ddit;
  dimension.erase(ddit);
  dimension_sizes.erase(dimension_sizes.begin()+n-1);
  if (get_storage()==1) sample_data_flag=true;
  return true;
}




Boolean DataDescriptor::is_sample_data() const {
  return sample_data_flag;
}
 
Integer DataDescriptor::get_n_dimensions() const {
  return n_dimensions;
}

Integer DataDescriptor::get_storage() const {
  if (n_dimensions==0) return 1;
  else return accumulate(dimension_sizes.begin(), dimension_sizes.end(), 1, multiplies<int>());
}

const DimensionDescriptor *DataDescriptor::get_dimension(const string &name) const {
  
  vector<DimensionDescriptor*>::const_iterator ddit;
  for (ddit=dimension.begin(); ddit!=dimension.end(); ++ddit) {
    if ((*ddit)->name_is(name)) break;
  }
  
  if (ddit!=dimension.end()) {
    (*ddit)->set_index(1+(ddit-dimension.begin()));
    return *ddit;
  } else
    return NULL;
}

const DimensionDescriptor *DataDescriptor::get_dimension(Integer index) const {

  if (index<=n_dimensions && index>0) {
    DimensionDescriptor *dd=dimension[index-1];
    dd->set_index(index);
    return dd;
  } else
    return NULL;
}

const DimensionDescriptor *DataDescriptor::get_inner_dimension() const {
  
  if (n_dimensions==0) return NULL;

  DimensionDescriptor *dd=dimension[0];
  dd->set_index(1);
  return dd;
}

const DimensionDescriptor *DataDescriptor::get_outer_dimension() const {

  if (n_dimensions==0) return NULL;
  
  DimensionDescriptor *dd=dimension[n_dimensions-1];
  dd->set_index(n_dimensions);
  return dd;
}



const vector<Integer> &DataDescriptor::get_dimension_sizes() const {
  return dimension_sizes;
}

const vector<Integer> &DataDescriptor::get_strides() const {
  // If the dimensions are [a_1, a_2 ... a_n] then the strides will be
  //   [a_2*a_3*...*a_n*1 ... a_(n-1)*a_(n)*1 a_(n)*1 1]
  Integer n=1;
  strides.clear();
  strides.push_back(1);  // Start of with [1]
  if (dimension_sizes.size()!=0) {
    for (vector<Integer>::const_reverse_iterator it=dimension_sizes.rbegin(); it!=dimension_sizes.rend()-1; ++it) {
      strides.insert(strides.begin(), n=(*it*n));
    }
  }
  return strides;
}

Boolean DataDescriptor::matches_shape_of(const DataDescriptor *this_dd) const {
  if (n_dimensions!=this_dd->n_dimensions) return false;
  vector<DimensionDescriptor*>::const_iterator ddit, ddit2;
  for (ddit=dimension.begin(), ddit2=this_dd->begin(); ddit!=dimension.end(); ++ddit, ++ddit2) {
    if (!(*ddit)->matches_size_of(*ddit2)) return false;
  }
  return true;
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: DimensionDescriptor		      			      */
/*									      */
/******************************************************************************/

DimensionDescriptor::DimensionDescriptor():name("\0"){};

DimensionDescriptor::DimensionDescriptor(const string &aname, const CTKVector &anaxis):name(aname), axis_values(anaxis), nPoints(anaxis.size()) {};

DimensionDescriptor::DimensionDescriptor(const DimensionDescriptor &this_desc):name(this_desc.name), axis_values(this_desc.axis_values), nPoints(this_desc.nPoints) { index=this_desc.index;}

DimensionDescriptor &DimensionDescriptor::operator=(const DimensionDescriptor &this_descriptor) {
  if (&this_descriptor!=this) {
    name=this_descriptor.name;
    axis_values=this_descriptor.axis_values;
    nPoints=this_descriptor.nPoints;
  }
  return *this;
}


const string&  DimensionDescriptor::get_name() const {
  return name;
}

Integer DimensionDescriptor::get_index() const {
  return index;
}

Integer DimensionDescriptor::size() const {
  return nPoints;
}

const CTKVector &DimensionDescriptor::get_axis() const {
  return axis_values;
}

Boolean DimensionDescriptor::name_is(const string &aname) const {
  return (name==aname);
}

Boolean DimensionDescriptor::matches_size_of(DimensionDescriptor *dd) const {
  return nPoints==dd->nPoints;
}

void DimensionDescriptor::set_index(Integer n) {
  index=n;
}



/* End of ctk_data_descriptor.cpp */
