/******************************************************************************/
/*									      */
/*	ctk_decoder.cpp	 	       			              */
/*									      */
/*	Support for Viterbi decoding                                                */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		         	      */
/*									      */
/******************************************************************************/
 /******************************************************************************/
/*									      */
/*	CLASS NAME: DecoderInfo 	              	       	              */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <cmath>
#include <cassert>

#include <vector>
#include <algorithm>
#include <functional>
#include <string>
#include <map>

#include <sstream>
#include <fstream>


#include "boost/compose.hpp"

#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_function_classes.hh"

#include "ctk_HMM.hh"
#include "ctk_reco.hh"
#include "ctk_parse.hh"
#include "ctk_lattice.hh"
#include "ctk_decoder.hh"
#include "ctk_group.hh"

#if defined (_DEBUG)
static int global_npushes = 0;
static int global_npulls = 0;
static int global_ncopy = 0;
static int global_nopeq = 0;
#endif

/******************************************************************************/
/*									      */
/*	CLASS NAME: GroupHypothesis            	         	              */
/*									      */
/******************************************************************************/

inline bool comp_group_hypotheses(const GroupHypothesis &d1, const GroupHypothesis &d2) { return d1.masked_name > d2.masked_name; }

GroupHypothesis::GroupHypothesis() {
  groups.resize(0);
  groups.push_back(-1);
  unique_name="_";
  masked_name="_";
  slot=0;
}

GroupHypothesis::GroupHypothesis(const string &a_group_record) {
  set_group_record(a_group_record);
  unique_name=a_group_record;
  masked_name=unique_name;
  slot=0;
}

GroupHypothesis::GroupHypothesis(const GroupHypothesis &hyp):groups(hyp.groups),unique_name(hyp.unique_name), masked_name(hyp.masked_name), slot(hyp.slot) {
};

GroupHypothesis &GroupHypothesis::operator=(const GroupHypothesis &gh) {
  if (&gh==this) return *this;
  groups=gh.groups;
  unique_name=gh.unique_name;
  masked_name=gh.masked_name;
  slot=gh.slot;
  return *this;
}

// Construct the missing data mask for the current group hypothesis given the group mask and the
// input missing data mask
// Mapping:  active groups -> inmask value , 0 and inactive groups -> 0
void GroupHypothesis::make_discrete_mask(const vector<Integer> &ingroup_mask, const vector<Float> &inmask, vector<Float> &outmask, bool inverted) {
  
  Integer group;
  vector<Float>::iterator op;
  vector<Float>::const_iterator ip;

  vector<Integer> group_mask = ingroup_mask;
  
  ip=inmask.begin();
  op=outmask.begin();
  for (vector<Integer>::iterator gp=group_mask.begin(); gp!=group_mask.end(); ++gp, ++ip, ++op) {
    if (*gp==0) {
      *op=0;     //  gp==0  => Background
      *gp=-99;
    }
  }

  for (UInteger i=0; i<groups.size(); ++i) {
    group=groups[i];
    ip=inmask.begin();
    op=outmask.begin();
    
    for (vector<Integer>::iterator gp=group_mask.begin(); gp!=group_mask.end(); ++gp, ++ip, ++op) {
      if (group==*gp) {
	*op=(inverted==false?*ip:0);
	*gp=-99;
      }
    }
  }

  ip=inmask.begin();
  op=outmask.begin();
  for (vector<Integer>::iterator gp=group_mask.begin(); gp!=group_mask.end(); ++gp, ++ip, ++op) {
    if (*gp!=-99) *op=(inverted==false?0:*ip);
  }

}


// Construct the missing data mask for the current group hypothesis given the group mask and the
// input missing data mask
// Mapping: 0 and active groups -> inmask value;  inactive groups -> 1-inmask value
void GroupHypothesis::make_soft_mask(const vector<Integer> &ingroup_mask, const vector<Float> &inmask, vector<Float> &outmask, bool inverted) {
  
  Integer group;
  vector<Float>::iterator op;
  vector<Float>::const_iterator ip;

  vector<Integer> group_mask = ingroup_mask;
  
  ip=inmask.begin();
  op=outmask.begin();
  for (vector<Integer>::iterator gp=group_mask.begin(); gp!=group_mask.end(); ++gp, ++ip, ++op) {
    if (*gp==0) {
      *op=*ip;     //  gp==0  => Background
      *gp=-99;
    }
  }

  for (UInteger i=0; i<groups.size(); ++i) {
    group=groups[i];
    ip=inmask.begin();
    op=outmask.begin();
    
    for (vector<Integer>::iterator gp=group_mask.begin(); gp!=group_mask.end(); ++gp, ++ip, ++op) {
      if (group==*gp) {
	*op=(inverted==false?*ip:(1.0-*ip));
	*gp=-99;
      }
    }
  }

  ip=inmask.begin();
  op=outmask.begin();
  for (vector<Integer>::iterator gp=group_mask.begin(); gp!=group_mask.end(); ++gp, ++ip, ++op) {
    if (*gp!=-99) *op=(inverted==false?(1.0-*ip):*ip);  
  }

}



void GroupHypothesis::make_masked_name(const Group &dead_group) {
  //  masked_name=unique_name;
  masked_name[dead_group.number()]='X';
}

void GroupHypothesis::make_masked_name_all_groups_dead() {
  //  masked_name=unique_name;
  for (unsigned int i=1; i<masked_name.size(); i++)
    masked_name[i]='X';
}

void GroupHypothesis::set_group_record(const string &group_record_string) {
  groups.resize(0);
  groups.push_back(-1);
  for (UInteger i=0, i_end=group_record_string.size(); i!=i_end; ++i) {
    if (group_record_string[i]=='1') groups.push_back(i);
  }
}

void GroupHypothesis::add_group(const Group &group, Boolean mask_flag) {

  // Append a 1 or a 0 to the unique group hypothesis name string

  string new_bool_string=(mask_flag?string("1"):string("0"));
  unique_name+=new_bool_string;
  masked_name+=new_bool_string;
  
  if (mask_flag)
    groups.push_back(group.number());
}


// Friend function for displaying GroupHypothesis member variables
ostream& operator<< (ostream& out, const GroupHypothesis& x) {
  // out << unique_name << "\n";
  out << "GroupHypothesis: \n";
  out << "Member groups: ";
  copy(x.groups.begin(), x.groups.end(), ostream_iterator<int>(out, " "));
  out << "\nSlot = " << x.slot << "\n";
  out << "Unique name = " << x.unique_name << "\n";
  out << "Masked name = " << x.masked_name << "\n";
  return out;
}



/******************************************************************************/
/*									      */
/*	CLASS NAME: Token  	              	         	              */
/*									      */
/******************************************************************************/

Token::Token():wrp(NULL), score(-numeric_limits<float>::max()),group_record("_\0"), labelp(), state_trace(0) {
};   // Tokens are intialized with huge -ve score and empty group record

// Copy constructor for token
Token::Token(const Token &atoken):wrp(atoken.wrp), score(atoken.score), group_record(atoken.group_record), labelp(atoken.labelp), state_trace(atoken.state_trace) { 
#if defined _DEBUG
  ++global_ncopy;      // FOR debugging - counts token copies 
#endif
};

// Assignment operator for token
Token &Token::operator=(const Token &atoken) {
  if (this!=&atoken) {
    wrp=atoken.wrp;
    score=atoken.score;
    group_record=atoken.group_record;
    labelp=atoken.labelp;
    state_trace=atoken.state_trace; 
#if defined _DEBUG
    ++global_nopeq;      // FOR debugging - counts all token copies  
#endif
  }
  return *this;
}


// An ammended assignment - i.e. the score and emmitted status are adjusted
void Token::ammend(const Token &atoken, HMMFloat ascore, const string *alabelp) {
  wrp=atoken.wrp;
  group_record=atoken.group_record;
  state_trace=atoken.state_trace;
  score=ascore;
  labelp=alabelp;
#if defined _DEBUG
  ++global_nopeq;      // FOR debugging - counts all token copies  
#endif
}


// Extend word record of the token  - for 1-best decoding
void Token::extendWordRecord(int frame, Float word_creation_penalty) {
  if (labelp!=NULL) {
    wrp=new WordRecord(frame, *this);  
    score-=word_creation_penalty;
    labelp=NULL;
  }
  state_trace.resize(0); 
}  

// Extend word record of the token  - records all tokens leaving state to allow N-best decoding
void Token::extendWordRecord(int frame, Float word_creation_penalty,  const list<Token> &exit_tokens) {
  // If token has come from an emitting node then extend Word Record
  if (labelp!=NULL) { 
    wrp=new WordRecord(frame, *this, exit_tokens); 
    score-=word_creation_penalty;
    labelp=NULL;
  }
  state_trace.resize(0); 
} 

const string* Token::getLabelp() const {
  return labelp;
}


void Token::addGroup(Boolean flag){
  if (isValid())
    group_record+=(flag?string("1"):string("0"));
}

void Token::activate(){score=0.0;}
void Token::deactivate(){score=-numeric_limits<float>::max();}



// Friend function for writing GroupHypothesis class details
ostream& operator<< (ostream& out, const Token& x) {
  out << "Token: \n";
  out << "   score = " << x.score << "\n";
  out << "   group_record = " << x.group_record << "\n";
  out << "   state_trace = ";
  copy(x.state_trace.begin(), x.state_trace.end(), ostream_iterator<int>(out, " "));
  out << "\n";
  out << "   wrp = " << x.wrp << "\n";
  return out;
}



/******************************************************************************/
/*									      */
/*	CLASS NAME: WordRecord                 	         	              */
/*									      */
/******************************************************************************/

list<WordRecord *> WordRecord::record_list = list<WordRecord *>();  // Definition for static member variable

WordRecord::WordRecord(int aframe, Token best, const list<Token> &all):frame(aframe), best_token(best), all_tokens(all) {
  record_list.push_back(this);
}

WordRecord::WordRecord(int aframe, Token best): frame(aframe), best_token(best), all_tokens() { 
  //  best_token.clearGroupRecord();
  record_list.push_back(this);
  all_tokens.clear();
}

void WordRecord::destroyAllWordRecords() {
#if defined (_DEBUG)
  cerr << "Destroying " << record_list.size() << " word records\n";
  cerr << "Total pushes = " << global_npushes << "; Total pulles = " << global_npulls << "\n";
  cerr << "Total token copies = " << global_ncopy << ";Total token op=() = " << global_nopeq << "\n";
#endif
  sequence_delete(record_list.begin(), record_list.end());
  record_list.resize(0);
}

const WordRecord *WordRecord::getPrevWordRecord() const {return best_token.getWRP();}

const string *WordRecord::getLabelp() const {return best_token.getLabelp();}

 

vector<RecoHypothesis*> WordRecord::backTrace(vector<RecoHypothesis*> &hyps, HMMFloat threshold, list<string> labelsofar, HMMFloat scoresofar, HMMFloat relative_scoresofar, list<int> boundsofar, list<vector<int> > statesofar, string groups) const {

  list<HMMFloat> exit_scores;

  list<Token> all_token_list;
  if (all_tokens.size()==0)
    // If the all token list is empty then Nbest decoding was turned off. 
    // So put the 1 best token on the all token list
    all_token_list.push_back(best_token);
  else
    all_token_list=all_tokens;

  list<Token>::const_iterator tp_end=all_token_list.end();
  for (list<Token>::const_iterator tp=all_token_list.begin(); tp!=tp_end; ++tp) {

    exit_scores.push_back(tp->getScore());
  }  
  
  list<HMMFloat>::iterator esp = max_element(exit_scores.begin(), exit_scores.end());
  
  HMMFloat maxscore = 0.0;

  if (esp!=exit_scores.end()) {
    maxscore=*esp;
  } else {
    cerr << "WordRecord::backTrace can't find max exit score\n";
    exit(-1);
  }
				    
  list<Token>::const_iterator tp = all_token_list.begin();
  esp = exit_scores.begin();
  while (tp!=tp_end) {

    //    HMMFloat newscoresofar =scoresofar+tp->getScore();;
    HMMFloat newscoresofar=(scoresofar==0.0?tp->getScore():scoresofar);

    HMMFloat newrelative_scoresofar= relative_scoresofar+tp->getScore()-maxscore;

    if (newrelative_scoresofar>=threshold) {
      
      
      list<string> newlabelsofar= labelsofar;
      const string *labelp=tp->getLabelp();
      if (labelp!=NULL) 
	newlabelsofar.push_front(*labelp);
      boundsofar.push_front(frame);

      // Insert the HMMs state trace at the front of the state trace list
      statesofar.push_front(tp->getStateTrace());
      bool statesofar_ok=false;
      if (statesofar.begin()->size()==0) {
	statesofar.pop_front();
	statesofar_ok=true;
      }
      
      const WordRecord *wrp=tp->getWRP();
      if (wrp!=NULL) { 
	wrp->backTrace(hyps, threshold, newlabelsofar, newscoresofar, newrelative_scoresofar, boundsofar, statesofar, groups);
      } else {
	RecoHypothesis *newhyp = new RecoHypothesis(newlabelsofar, scoresofar+newrelative_scoresofar, boundsofar, statesofar, groups);
	hyps.push_back(newhyp);
      }

      // Restore boundsofar and statesofar to their value before `if (newscoresofar>=threshold)' 
      if (statesofar_ok==false)
	statesofar.pop_front();
      boundsofar.pop_front();
    }
    ++esp;
    ++tp;
  }
  return hyps;
}

// Friend function for writing WordRecord class details
ostream& operator<< (ostream& out, const WordRecord& x) {
  out << "WordRecord: \n";
  out << "   frame = " << x.frame << "\n";
  out << "   best_token = " << x.best_token << "\n";
  out << "   all_tokens = (" << x.all_tokens.size() << " tokens)\n";
  return out;
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: Node  	              	         	              */
/*									      */
/******************************************************************************/

// Static variables
int Node::token_slot = 0;           // Initialise the token slot to 0
int Node::new_node_number = 0;      // Initialise new node number
bool Node::PRUNING = false;

HMMFloat Node::MAXOBSERVEDPROB = -numeric_limits<float>::max();
HMMFloat Token::MAXVALIDPROB = -numeric_limits<float>::max();

Node::~Node(){
  while (push_transitions.size()>0) 
    delete push_transitions.back();

  while (push_transitions_received.size()>0) 
    delete push_transitions_received.back();
  
  while (pull_transitions_supplied.size()>0) 
    delete pull_transitions_supplied.back();
  
}

void Node::activate() {

#if defined (_DEBUG)
  assert(token_slot==0);
#endif

  token[token_slot].activate(); // Set token prob to 0
} 

void Node::deactivate() {

#if defined (_DEBUG)
  assert(token_slot==0);
#endif
  
  token[token_slot].deactivate(); // Set token prob to a large -ve number
}

void Node::mergeTokens(const vector<vector<int> >&merge_slots) {

  // Replace token list with merged list of tokens.
  
  vector<Token> merged_tokens;
  // Note slot numbers here are indexed from 0.
  for(vector<vector<int> >::const_iterator vvip = merge_slots.begin(), vvip_end=merge_slots.end(); vvip!=vvip_end; ++vvip) {
    //    for(vector<int>::const_iterator vipx = vvip->begin(), vip_end=vvip->end(); vipx!=vip_end; ++vipx) 
    //     cerr << *vipx << " ";
    //   cerr << "\n";
    Token winning_token=token[*(vvip->begin())];
    for(vector<int>::const_iterator vip = vvip->begin()+1, vip_end=vvip->end(); vip!=vip_end; ++vip) {
      const Token &next_token=token[*vip];
      if (next_token.getScore()>winning_token.getScore()) {
	winning_token=next_token;
      }
    }
    merged_tokens.push_back(winning_token);
  }

  token=merged_tokens;
}

// Reset node - must be called on all nodes between decodings
void Node::reset() {
  sequence_delete(updated_token_pointers.begin(), updated_token_pointers.end());
  updated_token_pointers.clear();
  token.clear();
  token.resize(1);
  token_copy=Token();
}


// Add a forward connection to another node
PushTransition *Node::connectTo(Node *to_node, HMMFloat prob) {
  PushTransition *pushp=new PushTransition(this, to_node, prob);
  push_transitions.push_back(pushp);
  to_node->addPushTransitionReceived(pushp);
  return pushp;
}


// New group started - double up tokens and add group backtrace info
void Node::newGroup() {

  int N=token.size();

  // Duplicate tokens

  token.reserve(N*2);
  copy(token.begin(), token.end(), back_inserter(token));

  // Group Info can consume a lot of memory - so it is only recorded when requested.
  if (Decoder::requiresGroupInfo()) {
    // In one set the new group is assumed to be background
    for(vector<Token>::iterator tp = token.begin(), tp_end=token.begin()+N; tp!=tp_end; ++tp)
      tp->addGroup(false);
    
    // // In the other set the new group is assumed to be speech
    for(vector<Token>::iterator tp = token.begin()+N, tp_end=token.end(); tp!=tp_end; ++tp)
      tp->addGroup(true);
  }
}

// push tokens to next state
void Node::pushToken() {
  if (hasValidToken()) {
    Token *tokenp=&token[token_slot];
    for_each(push_transitions.begin(), push_transitions.end(), bind2nd(mem_fun(&PushTransition::push), tokenp));
  }

}

bool compareUpdatedTokens(const UpdatedToken *utp1, const UpdatedToken *utp2) {
  return (utp1->newprob_<utp2->newprob_);
}

void Node::realiseUpdatedTokens() {
  new_tokens.resize(0);
  list<const UpdatedToken*>::iterator utp_end=updated_token_pointers.end();
  for (list<const UpdatedToken*>::iterator utp=updated_token_pointers.begin(); utp!=utp_end; ++utp) {
    Token token(*((*utp)->tp_));
    token.setScore((*utp)->newprob_);
    token.setLabelp((*utp)->labelp_);
    new_tokens.push_back(token);
  }

}

// Replace token with best of new_tokens and clear new_tokens
void Node::compareTokens(bool debug) {

  if (debug) {
    cerr << "NENode::compareTokens: " << updated_token_pointers.size();
    if (updated_token_pointers.size()>0)
      cerr << " " << (*updated_token_pointers.begin())->newprob_;
    if (updated_token_pointers.size()>1) 
      cerr << " " << (*(++updated_token_pointers.begin()))->newprob_;
    cerr << "\n";
  }

  list<const UpdatedToken*>::iterator utp=max_element(updated_token_pointers.begin(), updated_token_pointers.end(), compareUpdatedTokens);

  if (utp!=updated_token_pointers.end()) {
    if (PRUNING) {
      if ((*utp)->newprob_>MAXOBSERVEDPROB) MAXOBSERVEDPROB=(*utp)->newprob_;
    }
  
    token_copy.ammend(*((*utp)->tp_), (*utp)->newprob_, (*utp)->labelp_);
    // Update tokens state list at this point
    if (record_state_path==true)
      recordStateToToken(token_copy);
  } else {
    token_copy.deactivate();
  }

  //  cerr << "token copy: " << token_copy.getScore() << "\n";
}

void Node::updateToken() {
  if (token_copy.isValid())
    token[token_slot]=token_copy;
  else
    token[token_slot].deactivate();
}

void Node::clearUpdatedTokenPointers() {
  sequence_delete(updated_token_pointers.begin(), updated_token_pointers.end());    
  updated_token_pointers.resize(0); 
}

void Node::reattachTransitions(NonEmittingNode &redirect_node) {

  // Redirect all transitions referencing this node so that they reference the redirect node
  
  if (&redirect_node==dynamic_cast<NonEmittingNode*>(this)) return;
 
  while (pull_transitions_supplied.size()>0)
    pull_transitions_supplied.back()->changeDistalNode(&redirect_node);
  
  while (push_transitions_received.size()>0)
    push_transitions_received.back()->changeDistalNode(&redirect_node);

  while (push_transitions.size()>0)
    push_transitions.back()->changeProxalNode(&redirect_node);

}

void Node::removePushTransition(PushTransition *trans) {
  //    assert(find(push_transitions.begin(), push_transitions.end(), trans)!=push_transitions.end());
  push_transitions.erase(remove(push_transitions.begin(), push_transitions.end(), trans), push_transitions.end());
}

void Node::removePushTransitionReceived(PushTransition *trans) {
  //    assert(find(push_transitions_received.begin(), push_transitions_received.end(), trans)!=push_transitions_received.end());
  push_transitions_received.erase(remove(push_transitions_received.begin(), push_transitions_received.end(), trans), push_transitions_received.end());
}

void Node::removePullTransitionSupplied(PullTransition *trans) {
  //    assert(find(pull_transitions_supplied.begin(), pull_transitions_supplied.end(), trans)!=pull_transitions_supplied.end());
  pull_transitions_supplied.erase(remove(pull_transitions_supplied.begin(), pull_transitions_supplied.end(), trans), pull_transitions_supplied.end());
}
  

ostream &NonEmittingNode::display(ostream& out) const {
  Node::display(out);
  out << "Pull Transitions: " << pull_transitions.size() << endl;
  
  for (list<PullTransition*>::const_iterator pullp = pull_transitions.begin(); pullp!=pull_transitions.end(); ++pullp)
    (*pullp)->display(out);

  return out;
}


ostream &Node::display(ostream& out) const {
  out << "Node number: " << node_number  << endl;

  out << "Pull transitions supplied: "  << pull_transitions_supplied.size()<< endl;
  for (list<PullTransition*>::const_iterator pullp = pull_transitions_supplied.begin(); pullp!=pull_transitions_supplied.end(); ++pullp)
    (*pullp)->display(out);

  out << "Push transitions receieved: "  << push_transitions_received.size() << endl;
  for (list<PushTransition*>::const_iterator pushp = push_transitions_received.begin(); pushp!=push_transitions_received.end(); ++pushp)
    (*pushp)->display(out);

  
  out << "Push transitions: " << push_transitions.size() << endl;
  for (list<PushTransition*>::const_iterator pushp = push_transitions.begin(); pushp!=push_transitions.end(); ++pushp)
    (*pushp)->display(out);

  return out;
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: NonEmittingNode  	              	      	              */
/*									      */
/******************************************************************************/

// Construct a node by merging a set of nodes
NonEmittingNode::NonEmittingNode(list<NonEmittingNode*> &node_loop):Node(false) {

  // Just set the WCP to the average for the merging nodes
  word_creation_penalty=0.0;

  // Nbest list formation is inactive until explicitly activated
  save_nbest_info=false;

  flag=false;

  word_end_node=false;
  list<NonEmittingNode*>::iterator nenp_end=node_loop.end();
  for (list<NonEmittingNode*>::iterator nenp=node_loop.begin(); nenp!=nenp_end; ++nenp) { 
    word_creation_penalty+=(*nenp)->getWordCreationPenalty();
    word_end_node=word_end_node||(*nenp)->isWordEndNode();
  }
  
  word_creation_penalty/=node_loop.size();
  
  // Reattach all transitions in the loop so that they point to the new node
  for (list<NonEmittingNode*>::iterator nenp=node_loop.begin(); nenp!=nenp_end; ++nenp) {
    (*nenp)->reattachTransitions(*this);
  }

}

NonEmittingNode::~NonEmittingNode(){
  while (pull_transitions.size()>0)
    delete pull_transitions.back();
}

// Extend the word record for this token
void NonEmittingNode::extendWordRecord(int frame) {
  if (isWordEndNode()) {
    if (token_copy.isValid()) {
      if (save_nbest_info) {
	realiseUpdatedTokens();
	token_copy.extendWordRecord(frame, word_creation_penalty, new_tokens);
      } else {
	token_copy.extendWordRecord(frame, word_creation_penalty);
      }
    }
  }
}

void NonEmittingNode::reattachTransitions(NonEmittingNode &redirect_node) {

  Node::reattachTransitions(redirect_node);
  
  // Redirect all transitions referencing this node so that they reference the redirect node
  
  if (&redirect_node==dynamic_cast<NonEmittingNode*>(this)) return;
  
  while (pull_transitions.size()>0)
    pull_transitions.back()->changeProxalNode(&redirect_node);


}

// Add a pull transition from an emitting node
PullTransition *NonEmittingNode::connectFrom(Node *from_node, HMMFloat prob) {
  PullTransition *pullp=new PullTransition(this, from_node, prob);
  pull_transitions.push_back(pullp);
  from_node->addPullTransitionSupplied(pullp);
  return pullp;
}

// Pull tokens from previous states 
void NonEmittingNode::pullToken() {
  
  list<PullTransition*>::iterator pullp_end=pull_transitions.end();
  for (list<PullTransition*>::iterator pullp = pull_transitions.begin(); pullp!=pullp_end; ++pullp)
    if ((*pullp)->hasValidToken())  
      receiveToken((*pullp)->pull());
      
}

bool NonEmittingNode::recursiveNENLoopSearch(list<NonEmittingNode*> &node_loop) {
  if ((node_loop.size()>0) && (*node_loop.begin()==this)) return (node_loop.size()>1);

  // If we have come back to a node already visited (but not the start of the loop) then
  // give up 
  if (find(node_loop.begin(), node_loop.end(), this)!=node_loop.end())
    return false;
  
  node_loop.push_back(this);
  
  list<PullTransition*>::const_iterator pullp_end=pull_transitions.end();
  for (list<PullTransition*>::const_iterator pullp=pull_transitions.begin(); pullp!=pullp_end; ++pullp) {
    if ((*pullp)->recursiveNENLoopSearch(node_loop)) return true;
  }
  node_loop.pop_back();
  return false;
}

// Flag all nodes pointed to by from transitions
void NonEmittingNode::flagFromNodes() {
  for_each(pull_transitions.begin(), pull_transitions.end(), mem_fun(&PullTransition::flagNode));
}


// Friend function for writing NEN class details
ostream& operator<< (ostream& out, const NonEmittingNode& x) {

  out << "NonEmittingNode:" << endl;
  x.display(out);
  
  out << "Pull transitions" << endl;

  // // How do you use this form for a list of consts? JON
  // copy(x.pull_transitions.begin(), x.pull_transitions.end(), ostream_iterator<const PullTransition>(out));
  for (list<PullTransition*>::const_iterator pullp=x.pull_transitions.begin(); pullp!=x.pull_transitions.end(); ++pullp) 
    (*pullp)->display(out);

  out << "word_creation_penalty = " << x.word_creation_penalty  << endl;

  return out;
}

bool NonEmittingNode::bypassIfRedundant() {
  // Returns true if node if redundant else returns false
  
  //  if (nPullTransitions()!=1) return false;
  if (nPullTransitions()>1) return false;
  
  if (nPullTransitions()==0) return true;
  
  if ((nPullTransitionsSupplied()+nPushTransitions())==0) return true;
  
  if (nPushTransitions()==1 && nPullTransitionsSupplied()==0 && nPushTransitionsReceived()==0) {
    // NEN <- (NEN) -> X   =>  NEN -> X
    bool pull_from_EN = pull_transitions.back()->getNode()->isEmitting();
    if (pull_from_EN && word_end_node) return false;
    Node *NEN = pull_transitions.back()->getNode();
    Node *X = push_transitions.back()->getNode();
    assert(NEN!=NULL);

    NEN->connectTo(X, 1.0); 
    // cerr << "Bypassing node " << getID() << ": NEN(" << NEN->getID() << ") -> X()\n";
    return true;
  }
  
  if (nPushTransitions()==0 && nPullTransitionsSupplied()==1 && nPushTransitionsReceived()==0) {
    // X <- (NEN) <- NEN  =>  X <- NEN
    Node *X = pull_transitions.back()->getNode();
    NonEmittingNode *NEN = dynamic_cast<NonEmittingNode*>(pull_transitions_supplied.back()->getBaseNode());
    string label = pull_transitions.back()->getLabel();
    assert(NEN!=NULL);
    // Construct bypass transition remembering to include any label that may have been on the pull transition that will be removed
    PullTransition *ptp=NEN->connectFrom(X, 1.0);
    ptp->setLabel(label);
    // cerr << "Bypassing node " << getID() << ": X() <= NEN(" << NEN->getID() << ")\n";
    return true;
  }
  
  return false;
}

void NonEmittingNode::removeSelfTransitions() {

  list<PullTransition*>::iterator new_end=remove_if(pull_transitions.begin(), pull_transitions.end(), mem_fun(&Transition::selfTransition));
  //  sequence_delete(new_end, pull_transitions.end());  
  pull_transitions.erase(new_end, pull_transitions.end());

  list<PushTransition*>::iterator new_end2=remove_if(push_transitions.begin(), push_transitions.end(), mem_fun(&Transition::selfTransition));
  // sequence_delete(new_end2, push_transitions.end());
  push_transitions.erase(new_end2, push_transitions.end());
  
}


void  NonEmittingNode::removePullTransition(PullTransition *trans) {
  //    assert(find(pull_transitions.begin(), pull_transitions.end(), trans)!=pull_transitions.end());
  pull_transitions.erase(remove(pull_transitions.begin(), pull_transitions.end(), trans), pull_transitions.end());
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: EmittingNodeBase  	      	         	              */
/*									      */
/******************************************************************************/

EmittingNodeBase::EmittingNodeBase():Node(true) {}

void EmittingNodeBase::activateStatePathRecording(bool x) {record_state_path=x;}

/******************************************************************************/
/*									      */
/*	CLASS NAME: EmittingNode  	      	         	              */
/*									      */
/******************************************************************************/

EmittingNode::EmittingNode(const HMMState * a_state, int a_state_num):EmittingNodeBase(), label(), has_label(0), state(a_state), state_num(a_state_num) {};

void EmittingNode::addEmissionProb(){
  token[token_slot].addProb(state->get_prob());
}

// Friend function for writing EmittingNode class details
ostream& EmittingNode::display(ostream& out) const {
  out << "EmittingNode:" << endl;
  out << "Label = " << getLabel()  << endl;

  return out;
}

void EmittingNode::setLabel(const string &alabel) {
  label=alabel;
  has_label=true;
}

const string &EmittingNode::getLabel() const {return label;}


/******************************************************************************/
/*									      */
/*	CLASS NAME: CompositeEmittingNode  	      	         	              */
/*									      */
/******************************************************************************/

CompositeEmittingNode::CompositeEmittingNode(const HMMState *astate1, int a_state_num1, const HMMState *astate2, int a_state_num2):EmittingNodeBase(), label1(), has_label1(0), state1(astate1), state_num1(a_state_num1), label2(), has_label2(0), state2(astate2), state_num2(a_state_num2) {};

void CompositeEmittingNode::addEmissionProb(){
  token[token_slot].addProb(state1->get_prob());
  token[token_slot].addProb(state2->get_prob());
}

// Friend function for writing CompositeEmittingNode class details
ostream &CompositeEmittingNode::display(ostream & out) const{
  out << "CompositeEmittingNode:" << endl;
  out << "state_num1 = " << state_num1  << endl;
  out << "state_num2 = " << state_num2  << endl;
  
  return out;
}

void CompositeEmittingNode::setLabel1(const string &alabel) {
  label1=alabel;
  has_label1=true;
}

void CompositeEmittingNode::setLabel2(const string &alabel) {
  label2=alabel;
  has_label2=true;
}

const string &CompositeEmittingNode::getLabel1() const {return label1;}
const string &CompositeEmittingNode::getLabel2() const {return label2;}

/******************************************************************************/
/*									      */
/*	CLASS NAME: Transition                 	         	              */
/*									      */
/******************************************************************************/
 
Transition::Transition(Node *abase_node, Node *anode, HMMFloat aprob): base_node(abase_node), node(anode), log_prob(aprob>0.0?log(aprob):-numeric_limits<float>::max()) {};

Transition::~Transition() {
};

ostream& Transition::display(ostream &out) const {
  out << "Transition: " << base_node->getNodeNumber() << " -- " << node->getNodeNumber() << " (with log_prob " << log_prob << ")\n";
  return out;
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: PullTransition                 	      	              */
/*									      */
/******************************************************************************/

PullTransition::~PullTransition(){
  node->removePullTransitionSupplied(this);
  NonEmittingNode* np=dynamic_cast<NonEmittingNode*>(base_node);
  if (np!=NULL)
    np->removePullTransition(this);
}

void  PullTransition::changeProxalNode(NonEmittingNode *anode) {
  if (base_node==anode) return;
  (dynamic_cast<NonEmittingNode*>(base_node))->pull_transitions.remove(this);
  base_node=anode;
  (dynamic_cast<NonEmittingNode*>(base_node))->pull_transitions.push_back(this);
}

void  PullTransition::changeDistalNode(Node *anode) {
  if (node==anode) return;
  node->pull_transitions_supplied.remove(this);
  node=anode;
  node->pull_transitions_supplied.push_back(this);
}

UpdatedToken *PullTransition::pull() const {
#if defined (_DEBUG)
  ++global_npulls;      // FOR debugging - counts all pulls
#endif
  const Token *tokenp = &node->getToken();
  //  cerr << "Pull transition "  << tokenp->getScore() << " " << log_prob << "\n";
  if (has_label)
    return new UpdatedToken(tokenp, tokenp->getScore()+log_prob, &label);
  else
    return new UpdatedToken(tokenp, tokenp->getScore()+log_prob);
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: PushTransition                            	              */
/*									      */
/******************************************************************************/

PushTransition::~PushTransition(){
  node->removePushTransitionReceived(this);
  base_node->removePushTransition(this);
}

void  PushTransition::changeProxalNode(Node *anode) {
  if (base_node==anode) return;
  base_node->push_transitions.remove(this);
  base_node=anode;
  base_node->push_transitions.push_back(this);
}

void  PushTransition::changeDistalNode(Node *anode) {
  if (node==anode) return;
  node->push_transitions_received.remove(this);
  node=anode;
  node->push_transitions_received.push_back(this);
}

void PushTransition::push(const Token *tokenp) const {
#if defined (_DEBUG)
  ++global_npushes;      // FOR debugging - counts all pushes
#endif  
  node->receiveToken(new UpdatedToken(tokenp, tokenp->getScore()+log_prob));
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: Decoder                	         	              */
/*									      */
/*****************************************************************************/
 
// Static variables
bool Decoder::group_recording_switch = true;           // Record group info

// Construct the node network for a simple loop grammar case
Decoder::Decoder(const SetOfHMMs *these_hmms, const string &prefix, const string &postfix, HMMFloat word_creation_penalty):network_start_node(0), network_end_node(0), hmms(these_hmms) {

  // Reset new node number - i.e. first node added to decoder will be numbered 0
  Node::resetNewNodeNumber();

  // Build network for loop grammar decoder from a set of HMM models 
  if (constructLoopGrammar(hmms, prefix, postfix)==CTK_FAILURE) {
    cerr << "Warning: Error during construction of decoder.\n";
  }

  // Configure network
  initialise();
  
  setWordCreationPenalty(word_creation_penalty);

}


// Construct the node network according to a grammar file
Decoder::Decoder(const SetOfHMMs *these_hmms,
		 HMMFloat word_creation_penalty,
		 const string &grammar_file,
		 const string &grammar_format)      // Valid values are SLF or EBNF
  :network_start_node(0), network_end_node(0), hmms(these_hmms) {
  
  // Reset new node number - i.e. first node added to decoder will be numbered 0
  Node::resetNewNodeNumber();

  if (grammar_format=="EBNF") {
    // Build network according to Extended Backus-Naur Form grammar file
    constructGrammarFromEBNFFile(grammar_file);
  } else if (grammar_format=="SLF") {
    // Build network according to Standard Lattice Format file
    constructGrammarFromSLFFile(grammar_file);
  } else {
    cerr << "Unrecognised grammar file format: " << grammar_format << "\n";
    return;
  }
  
  // Configure network
  initialise();
  
  setWordCreationPenalty(word_creation_penalty);

  // cout << *this;
}

Decoder::Decoder(const Decoder &decoder1, const Decoder &decoder2):network_start_node(0), network_end_node(0), hmms(NULL) {

  
  map<string, Node*> node_name_map;

  // Generate all the states
  
  multiplyOutNodes(decoder1.getEmittingNodesBegin(), decoder1.getEmittingNodesEnd(),
			   decoder2.getEmittingNodesBegin(), decoder2.getEmittingNodesEnd(),
			   emitting_nodes, node_name_map);
  
  multiplyOutNodes(decoder1.getEmittingNodesBegin(), decoder1.getEmittingNodesEnd(),
			      decoder2.getLinkNodesBegin(), decoder2.getLinkNodesEnd(),
			      link_nodes, node_name_map);
  
  multiplyOutNodes(decoder1.getLinkNodesBegin(), decoder1.getLinkNodesEnd(),
			      decoder2.getEmittingNodesBegin(), decoder2.getEmittingNodesEnd(),
			      link_nodes, node_name_map);
  
  multiplyOutNodes(decoder1.getLinkNodesBegin(), decoder1.getLinkNodesEnd(),
			      decoder2.getLinkNodesBegin(), decoder2.getLinkNodesEnd(),
			      link_nodes, node_name_map);

  // Generate all the push transitions
  
  multiplyOutPushTransitions(decoder1.getEmittingNodesBegin(), decoder1.getEmittingNodesEnd(),
			     decoder2.getEmittingNodesBegin(), decoder2.getEmittingNodesEnd(),
			     node_name_map);
  
  multiplyOutPushTransitions(decoder1.getLinkNodesBegin(), decoder1.getLinkNodesEnd(),
			     decoder2.getEmittingNodesBegin(), decoder2.getEmittingNodesEnd(),
			     node_name_map);

  multiplyOutPushTransitions(decoder1.getEmittingNodesBegin(), decoder1.getEmittingNodesEnd(),
			     decoder2.getLinkNodesBegin(), decoder2.getLinkNodesEnd(),
			     node_name_map);
  
  multiplyOutPushTransitions(decoder1.getLinkNodesBegin(), decoder1.getLinkNodesEnd(),
			     decoder2.getLinkNodesBegin(), decoder2.getLinkNodesEnd(),
			     node_name_map);

  // Generate all the pull transitions

  multiplyOutPullTransitions(decoder1.getLinkNodesBegin(), decoder1.getLinkNodesEnd(),
			     decoder2.getLinkNodesBegin(), decoder2.getLinkNodesEnd(),
			     node_name_map);
  
  multiplyOutPullTransitions(decoder1.getEmittingNodesBegin(), decoder1.getEmittingNodesEnd(),
			     decoder2.getLinkNodesBegin(), decoder2.getLinkNodesEnd(),
			     node_name_map);
  
  multiplyOutPullTransitions(decoder1.getLinkNodesBegin(), decoder1.getLinkNodesEnd(),
			     decoder2.getEmittingNodesBegin(), decoder2.getEmittingNodesEnd(),
			     node_name_map);

  // Set the start and end nodes
  
  Node *end_node = node_name_map[compositeNodeName(decoder1.getEndNode(), decoder2.getEndNode())];
  if (end_node==NULL) 
    cerr << "Cannot find composite end node\n";
  else
    setEndNode(dynamic_cast<NonEmittingNode*>(end_node));

  Node *start_node = node_name_map[compositeNodeName(decoder1.getStartNode(), decoder2.getStartNode())];
  if (start_node==NULL) 
    cerr << "Cannot find composite start node\n"; 
  else
    setStartNode(dynamic_cast<NonEmittingNode*>(start_node));
  
  // Configure network
  initialise();
  
  cerr << emitting_nodes.size() << " Emitting Nodes\n";
  cerr << link_nodes.size() << " Non Emitting Nodes\n";
  // Word creation penalty?
}

string Decoder::compositeNodeName(const Node *node1, const Node *node2) {
  ostringstream namestr(ostringstream::out);
  namestr << node1->getNodeNumber() << ":" << node2->getNodeNumber();
  return namestr.str();
}

Node *Decoder::newCompositeNode(EmittingNodeBase *np1, EmittingNodeBase *np2, list<EmittingNodeBase*> &emit_node) {

  CompositeEmittingNode *enp=new CompositeEmittingNode(np1->getHMMState(), np1->getStateNum(), np2->getHMMState(), np2->getStateNum());
  enp->setLabel1(np1->getLabel());
  enp->setLabel2(np2->getLabel());
  emit_node.push_back(enp);
  return enp;
}
  
Node *Decoder::newCompositeNode(Node *, Node *, list<NonEmittingNode*> &non_emit_node) {
  NonEmittingNode *nenp=new NonEmittingNode();
  non_emit_node.push_back(nenp);
  return nenp;
}

void Decoder::multiplyOutPullTransitions(NENit npbegin1, NENit npend1, NENit npbegin2, NENit npend2, map<string, Node*> &namemap) {

    NENit np1;
    NENit np2;
   
    for (np1=npbegin1; np1!= npend1; ++np1) {
      for (np2=npbegin2; np2!= npend2; ++np2) {
	//	string label= string("1:")+(*np1)->getLabel()+"+2:"+(*np2)->getLabel();
	//	cerr << "label2: = " << label << "\n";
	string to_namestr = compositeNodeName(*np1, *np2);
	NonEmittingNode *to_node= dynamic_cast<NonEmittingNode*>(namemap[to_namestr]);
	
	if (to_node!=NULL) {
	  PullTit pt1, pt2;
	  PullTit ptend1 = (*np1)->getPullTransitionsEnd();
	  PullTit ptend2 = (*np2)->getPullTransitionsEnd();
	  
	  for (pt1=(*np1)->getPullTransitionsBegin(); pt1!= ptend1; ++pt1) {
	    for (pt2=(*np2)->getPullTransitionsBegin(); pt2!= ptend2; ++pt2) {
	      EmittingNodeBase *enbp;
	      Node *fnp1, *fnp2;
	      string from_namestr=compositeNodeName(fnp1=(*pt1)->getNode(), fnp2=(*pt2)->getNode());
	      string label;
	      enbp = dynamic_cast<EmittingNodeBase*>(fnp1);
	      if (enbp!=NULL) {
		label+=string("+1:")+ enbp->getLabel();
	      }
	      enbp = dynamic_cast<EmittingNodeBase*>(fnp2);
	      if (enbp!=NULL) {
		label+=string("+2:")+ enbp->getLabel();
	      }
	      //	      cerr << "label = " << label << "\n";
	      Node *from_node=namemap[from_namestr];
	      if (from_node!=NULL) {
		//		cout << "Connecting node " << to_node->getNodeNumber() << " from " << from_node->getNodeNumber() << "\n";
		 PullTransition *ptp=to_node->connectFrom(from_node, 1.0);
		 ptp->setLabel(label);
	      } else {
		cerr << "failed to find composite from_node: " << from_namestr << "\n";
	      }
	    }
	  }
	} else {
	  cerr << "failed to find composite to_node: " << to_namestr << "\n";
	}
      }
    }
    
  }




//Destructor
Decoder::~Decoder() {

  cleanUp();

  sequence_delete(link_nodes.begin(), link_nodes.end());
  sequence_delete(emitting_nodes.begin(), emitting_nodes.end());

}

void Decoder::activateStatePathRecording(bool x) {
  state_path_recording_switch=x;
  for (list<EmittingNodeBase*>::iterator enp=emitting_nodes.begin(); enp!=emitting_nodes.end(); ++enp)
    (*enp)->activateStatePathRecording(x);
}

void  Decoder::setNBestListSize(int list_size) {
  nbest_list_size=list_size;
  setRequiresNBestTraceInfo(list_size>1);
}

void Decoder::setRequiresNBestTraceInfo(bool requires_nbest) {
  for_each(link_nodes.begin(), link_nodes.end(), bind2nd(mem_fun(&NonEmittingNode::activateNBestInfo), requires_nbest));
}

CTKStatus Decoder::constructLoopGrammar(const SetOfHMMs *loop, const string &prefix, const string &postfix) {
  // Build network for loop grammar decoder from a set of HMM models with optional prefix and postfix models

  CTKStatus status = CTK_SUCCESS;
  
  NonEmittingNode *s_node=addNEN();
  setStartNode(s_node);

  NonEmittingNode* nenp = insertNodeAt(s_node, 1, 0);

  // Add add optional prefix HMM in series
  if (prefix.size()!=0) {
    nenp=addHMMByNameAt(prefix, nenp);
  }
  
  NonEmittingNode* end_node = insertNodeAt(nenp, 1, 1);

  status = addHMMByNameBetween(loop->get_name_list(), nenp, end_node);
  
  // Add optional postfix HMM in series
  nenp=end_node;
  
  if (postfix.size()!=0) {
    nenp=addHMMByNameAt(postfix, nenp);
  }
    
  setEndNode(nenp);

  return status;
}

// Build network according to Standard Lattice Format file
void Decoder::constructGrammarFromSLFFile(const string &slf_file) {

  std::istream *istr;
  
  istr= new std::ifstream(slf_file.c_str());
   
  if ((*istr)==NULL) {
    cerr << "Cannot open SLF file: " << slf_file  << endl;
    throw(CTKError(__FILE__, __LINE__));
  }  
  
  Lattice  lattice(*istr);
  
  NonEmittingNode *snp=addNEN();
  setStartNode(snp);
  NonEmittingNode *enp=addNEN();
  setEndNode(enp);
  NodePair node_pair = NodePair(snp, enp);
  
  lattice.buildNetwork(*this, node_pair);
  
}


// Build network according to Extended Backus-Naur Form grammar file
void Decoder::constructGrammarFromEBNFFile(const string &grammar_file) {
  
  
  std::istream *istr;
  
  if (grammar_file[0]=='(') {
    // The grammar parameter contains the actual grammar directly
    istr= new std::istringstream(grammar_file.c_str());
  } else {
    // The grammar parameter is the name of a file containing the grammar
    istr= new std::ifstream(grammar_file.c_str());
   
    if ((*istr)==NULL) {
      cerr << "Cannot open grammar file: " << grammar_file  << endl;
      throw(CTKError(__FILE__, __LINE__));
    }
  }
  
  
  FiniteStateGrammar   fsg(*istr);
  
  NonEmittingNode *snp=addNEN();
  setStartNode(snp);
  NonEmittingNode *enp=addNEN();
  setEndNode(enp);
  NodePair node_pair = NodePair(snp, enp);
  
  fsg.buildNetwork(*this, node_pair);
}

void Decoder::setWordCreationPenalty(HMMFloat penalty) {  // Sets a word creation penalty for all word endings in the network
  for_each(link_nodes.begin(), link_nodes.end(), bind2nd(mem_fun(&NonEmittingNode::setWordCreationPenalty), penalty));
}



NonEmittingNode *Decoder::addNEN(bool word_end) {
  NonEmittingNode *node=new NonEmittingNode(word_end);
  link_nodes.push_back(node);
  return node;
}

const HMM *Decoder::getHMMByName(string hmm_label) const {
  return hmms->get_HMM_by_name(hmm_label); 
}

// Add a pair of NEN's with an HMM sitting between. Return the NEN's. 
NodePair Decoder::addHMMByName(string hmm_label) {

  NodePair node_pair;
  if ( hmms->get_HMM_by_name(hmm_label)!=NULL) {
    node_pair=addNodePair(false, false);
    addHMMByNameBetween(hmm_label, node_pair.start, node_pair.end);
  }
    
  return node_pair;
}


void Decoder::initialise() {
  // Called after construction to put network into a form suitable for the Viterbi token passing algorithm

  // Replace Non Emitting Nodes loops with a single NEN
  removeNENLoops();

  // Remove redundant non-emitting nodes   x <- 0 <- 0 -> x  =>  x <- 0 -> x
  simplifyNENs(); 

// Store Non Emitting Nodes in the order in which tokens pass through them
  // (this is necessary if there are direct connections between NENs)
  orderNENs();

 }



void Decoder::reset() {
  // Called before token passing begins. Puts network into good initial state.

  group_hypothesis.clear();
  group_hypothesis.resize(1);

  reset_current_group_hypothesis();

  Node::ResetMaxObservedProb();
  Token::ResetMaxValidProb();
  
  // reset all nodes in the network
  for_each(emitting_nodes.begin(), emitting_nodes.end(), mem_fun(&Node::reset));
  for_each(link_nodes.begin(), link_nodes.end(), mem_fun(&Node::reset));

  current_frame=0;
  
  network_start_node->activate();  // Load start token

  // Get start token into position
  if (network_start_node->nPullTransitionsSupplied() != 0) {
    for (list<NonEmittingNode*>::iterator nenp=link_nodes.begin(); nenp!=link_nodes.end(); ++nenp) {
      if (*nenp==network_start_node) continue;  // Don't pull the start node
      (*nenp)->pullToken();
      (*nenp)->compareTokens();
      (*nenp)->clearUpdatedTokenPointers();
      (*nenp)->updateToken();
    }
    //  network_start_node->deactivate();  // Clear start token
  }
  
  network_start_node->activate();  // Reactivate start token

#if defined (_DEBUG)
  displayStatusSummary(cerr);
#endif

}

void Decoder::nextFrame() {
  ++current_frame;
}

void Decoder::passTokens() {
  // Step tokens forward one frame

  for_each(emitting_nodes.begin(), emitting_nodes.end(), mem_fun(&Node::pushToken));

  for_each(link_nodes.begin(), link_nodes.end(), mem_fun(&Node::pushToken));

  for_each(emitting_nodes.begin(), emitting_nodes.end(), bind2nd(mem_fun(&Node::compareTokens),false));

  for_each(emitting_nodes.begin(), emitting_nodes.end(), mem_fun(&Node::clearUpdatedTokenPointers));

  for_each(emitting_nodes.begin(), emitting_nodes.end(), mem_fun(&Node::updateToken));

  for_each(emitting_nodes.begin(), emitting_nodes.end(), mem_fun(&EmittingNodeBase::addEmissionProb)); 

  // Pull tokens through non emitting nodes ready for next push
  // Note -- this assumes tokens are stored in an appropriate order
  for (list<NonEmittingNode*>::iterator nenp=link_nodes.begin(); nenp!=link_nodes.end(); ++nenp) {
    (*nenp)->pullToken();
    (*nenp)->compareTokens(false);
    (*nenp)->extendWordRecord(current_frame);
    (*nenp)->clearUpdatedTokenPointers();
    (*nenp)->updateToken();
  }
  
  if (Node::Pruning()) {
    Token::SetMaxValidProb(Node::GetMaxObservedProb()-pruning_beamwidth);
    //   adaptPruningBeamwidth(97.0);  // JON
    //    cerr << "PRUNING\n";
  }

  Node::ResetMaxObservedProb();

  //  displayStatusSummary(cerr); // JON
 
}
  
// Gently adapt pruning beamwidth to aim at a given target percentage of pruned tokens
void Decoder::adaptPruningBeamwidth(float target_percent) {

  // This currently doesn't work well - all the extra token coounting slows decoding by about 10% and the beam tends to get too small  - but the idea is sensible enough...
  
  int nvalid=countValidTokens(); // This is slow - would be quicker (but messier) to compute the count during token passing where token validity if checked the first time
  int nnodes =emitting_nodes.size()+link_nodes.size(); 
  float prune_percent=100.0*(nnodes-nvalid)/nnodes;

  // increase beamwidth if too many tokens are being pruned, decrease beamwidth
  // if not enough tokens are being pruned

  pruning_beamwidth *= 1.0+(prune_percent-target_percent)/10.0;

}

void Decoder::displayStatusSummary(ostream &out, bool NEN_details) {
  int nvalid=countValidTokens(); // This is slow - would be quicker (but messier) to compute the count during token passing where token validity if checked the first time
  out << "nValid = " << nvalid << "; nEmitting = " << emitting_nodes.size() << "; nNEN = " << link_nodes.size() << "\n";

  if (NEN_details) {
    for (list<NonEmittingNode*>::const_iterator nenp=link_nodes.begin(); nenp!=link_nodes.end(); ++nenp)
      (*nenp)->display(out);
  }
}


CTKStatus Decoder::backTrace(vector<RecoHypothesis*> &hyps, HMMFloat threshold) {
  
  sequence_delete(hyps.begin(), hyps.end());
  hyps.resize(0);

  if (!network_end_node->getToken().isValid()) {
    cout << "Warning: No valid tokens made it to the end state" << endl;
    return CTK_FAILURE;
  }
    
  HMMFloat scoresofar=0.0;
  HMMFloat relative_scoresofar=0.0;

  list<string> labelsofar;
  list<Integer> boundsofar;
  list<vector<int> > statesofar;

  statesofar.resize(0);

  string groups=(network_end_node->getToken()).getGroupRecord();
  //  cerr << group_hypothesis[1].
  (network_end_node->getToken()).getWRP()->backTrace(hyps,threshold, labelsofar, scoresofar, relative_scoresofar, boundsofar, statesofar, groups);
 
  int list_size=getNBestListSize();
  if (list_size>1) {
    sort(hyps.begin(), hyps.end(), compare_hypotheses);
  }

  if (hyps.size()>=(unsigned)list_size) {
    sequence_delete(hyps.begin()+list_size, hyps.end());
    hyps.resize(list_size);
  } else if (hyps.size()<(unsigned)list_size) {
    // Failed to retrieve nbest_decoding hypotheses - try again with a lower threshold
    if (threshold<MIN_BACK_TRACE_THRESHOLD_CONST) {
      cout << "Warning: Fewer than " << list_size << " hypotheses available." << endl;
    } else
      backTrace(hyps, threshold-NBEST_THRESHOLD_STEP_SIZE);
  }

  return CTK_SUCCESS;
}

// Find the emitting node that has the token with the highest score so far
// - (this token is 'best so far' and is not necessarily on the eventual best path...)

const EmittingNodeBase *Decoder::getTopEmittingNode() const {
  
  list<EmittingNodeBase*>::const_iterator enp=emitting_nodes.begin(), enp_end=emitting_nodes.end();

  EmittingNodeBase* best_node=*enp;
  float best_score=best_node->getToken().getScore();
  float score;
  
  for (; enp!=enp_end; ++enp) {
    score=(*enp)->getToken().getScore();
    if (score>best_score) {
      best_score=score;
      best_node=*enp;
    }
  }

  return best_node;
}


const EmittingNodeBase *Decoder::getInstantTopEmittingNode() const {
   
  list<EmittingNodeBase*>::const_iterator enp=emitting_nodes.begin(), enp_end=emitting_nodes.end();

  EmittingNodeBase* best_node=*enp;
  float best_score=(*enp)->getHMMState()->get_prob();
  float score;
  
  for (; enp!=enp_end; ++enp) {
    score=(*enp)->getHMMState()->get_prob();
    if (score>best_score) {
      best_score=score;
      best_node=*enp;
    }
  }

  return best_node;
}

void Decoder::displayLogProbs(ostream &outfile) const {
  for (list<EmittingNodeBase*>::const_iterator enp=emitting_nodes.begin(); enp!=emitting_nodes.end(); ++enp)
    outfile << (*enp)->getToken().getScore() << " ";
  outfile  << endl;
}


int Decoder::countValidTokens() const {
  return (
    count_if(emitting_nodes.begin(),emitting_nodes.end(),mem_fun(&EmittingNode::hasValidToken))+
    count_if(link_nodes.begin(),link_nodes.end(),mem_fun(&NonEmittingNode::hasValidToken))
    );
}



//Display network information for debugging
ostream& operator<< (ostream& out, const Decoder& x) {
  // copy (x.emitting_nodes.begin(), x.emitting_nodes.end(), ostream_iterator<const EmittingNodeBase>(out));
  
  for (list<EmittingNodeBase*>::const_iterator enp=x.emitting_nodes.begin(), enp_end=x.emitting_nodes.end(); enp!=enp_end; ++enp)
    (*enp)->display(out);

  copy (x.link_nodes.begin(), x.link_nodes.end(), ostream_iterator<const NonEmittingNode>(out));

  return out;
}


///// Private methods

void Decoder::setStartNode(NonEmittingNode *nenp) {
  if (network_start_node!=NULL) network_start_node->deactivate(); // Set prob to a large -ve number
  network_start_node=nenp;
  network_start_node->activate();  // Set probability 0
}

void Decoder::setEndNode(NonEmittingNode *nenp) {
  network_end_node=nenp;
}

NonEmittingNode *Decoder::insertNodeAt(NonEmittingNode *at_node, bool is_optional, bool is_repeatable) {
  NonEmittingNode *node=addNEN();

  if (is_optional) node->connectFrom(at_node, 1.0);
  if (is_repeatable) at_node->connectFrom(node, 1.0);

  return node;
}

NodePair Decoder::insertNodePairAt(Node *at_node, bool is_optional, bool is_repeatable) {
  NodePair new_node_pair = addNodePair(is_optional, is_repeatable);
  
  new_node_pair.start->connectFrom(at_node, 1.0);

  return new_node_pair;
}

NodePair Decoder::insertNodePairBetween(NodePair &node_pair, bool is_optional, bool is_repeatable) {
  NodePair new_node_pair = addNodePair(is_optional, is_repeatable);

  new_node_pair.start->connectFrom(node_pair.start, 1.0);
  node_pair.end->connectFrom(new_node_pair.end, 1.0);
  
  return new_node_pair;
}


NonEmittingNode *Decoder::addHMMByNameAt(const string &hmm, NonEmittingNode *at_node) {

  // Make hmm a list of length one and then call the more general form for adding a list of HMMs
  
  vector<string> some_hmms;
  some_hmms.push_back(hmm);
  return addHMMByNameAt(some_hmms, at_node);
}


NonEmittingNode *Decoder::addHMMByNameAt(const vector<string> &some_hmms, NonEmittingNode *at_node) {

  NonEmittingNode *end_node=addNEN();

  addHMMByNameBetween(some_hmms, at_node, end_node);
  
  return end_node;

}

CTKStatus Decoder::addHMMByNameBetween(const vector<string> &some_hmms, NonEmittingNode *start_node, NonEmittingNode *end_node) {
  
  PronunciationTree tree;
  buildPronunciationTree(tree, some_hmms);
  addPronunciationTreeBetween(tree, start_node, end_node);
  /*
  for(vector<string>::const_iterator hmmpp=some_hmms.begin(); hmmpp!=some_hmms.end(); ++hmmpp) {
    if (addHMMByNameBetween(*hmmpp, start_node, end_node)==CTK_FAILURE)
      return CTK_FAILURE;
  }
  */
  return CTK_SUCCESS;
  
}


// Add HMM between a given pair of nodes
CTKStatus Decoder::addHMMByNameBetween(string hmm_name, NonEmittingNode *start_node, NonEmittingNode *end_node) {
  
  string label=hmms->lookup_label(hmm_name);
  if (label.size()==0) {
    cerr << "SetOfHMMs::lookup_label - can not find label for HMM with name = " << hmm_name << "\n";
    exit(-1);
  }
  
  list<string> pronunciation = hmms->lookup_pronunciation(hmm_name);

  if (pronunciation.size()==0) return CTK_FAILURE;
    
  list<string>::iterator pp=pronunciation.begin();
  list<string>::iterator pp_end=pronunciation.end();
  --pp_end;

  NonEmittingNode *nen1 = start_node;

  for (; pp!=pp_end;++pp) {
    nen1=addBaseHMMByName(*pp, nen1, NULL, label);
    if (nen1==NULL) {
      cerr << "Can not find HMM named: " << *pp << "\n";
      return CTK_FAILURE;
    }
  }

  if (addBaseHMMByName(*pp, nen1, end_node, label)==NULL) {
    cerr << "Can not find HMM named: " << *pp << "\n";
    return CTK_FAILURE;
  }

  return CTK_SUCCESS;
}


// Look up the HMM for the base modelling unit (word, triphone, monophone...) with the given name and add it at or between two nodes
NonEmittingNode *Decoder::addBaseHMMByName(const string &hmm_name, NonEmittingNode *at_node, NonEmittingNode *end_node, const string &label) {
  
    const HMM *hmm = hmms->get_HMM_by_name(hmm_name);
    if (hmm==NULL) {
      cerr << "Can not find HMM named: " << hmm_name << "\n";
      return NULL;
    }

    if (end_node==NULL)
      end_node = addNEN(false /* not a word end node */);

    addBaseHMMBetween(hmm, at_node, end_node, label);
    return end_node;

}


// Add the HMM for the base modelling unit (word, triphone, monophone...)  at or between two nodes
void Decoder::addBaseHMMBetween(const HMM *hmm, NonEmittingNode *start_node, NonEmittingNode *end_node, const string &label) {

  vector<Node *> new_nodes;

  // Construct new nodes for each hmm state
  for (int state_num=0; state_num<hmm->get_num_states(); ++state_num) {
    if (hmm->get_state(state_num)->emits()) {
      EmittingNodeBase *enp;
      new_nodes.push_back(enp=new EmittingNode(hmm->get_state(state_num), state_num));
      enp->setLabel(label);
      emitting_nodes.push_back(enp);
    } else {
      new_nodes.push_back(addNEN(false /* not a word end node */)); 
    }
  }
  
  HMMFloat p;
  unsigned int nstates=hmm->get_num_states();
  
  // Deal with transitions among emitting nodes
  const Transitions &trans = hmm->getTrans();
  for (unsigned int i=1; i<=nstates; ++i) {
    Node *from_node=new_nodes[i-1];
    for (unsigned int j=1; j<=nstates; ++j) {
      if ((p=trans.get_transition_prob(i,j))!=0) {
	Node *to_node=new_nodes[j-1];
	if (to_node->isEmitting())
	  from_node->connectTo(to_node,p);
	else
	  ((NonEmittingNode*)to_node)->connectFrom(from_node,p);
      }
    }
  }
  
  for (unsigned int i=1; i<=nstates; ++i) {
    // Deal with connections from non-emitting start state to first HMM state
    if ((p=trans.get_transition_prob(0, i))!=0) {
      start_node->connectTo(new_nodes[i-1],p);
    }
    // Deal with connection to non-emitting end state from final HMM state
    if ((p=trans.get_transition_prob(i, nstates+1))!=0) {
      PullTransition* ptp=end_node->connectFrom(new_nodes[i-1],p);
      ptp->setLabel(label);
    }
  }
}

void Decoder::removeNENLoops() {
  // Replace loops of NENs with the equivalent single NEN:
  // 0. Starting at each NEN node in turn:
  //  1. Recursively traverse graph tagging each node visited
  //  2. Terminate when arriving at an EN or a tagged NEN
  //  3. If (terminating node = starting node) then loop detected.
  //  4. If loop detected a) merge all tagged nodes
  //                      b) restart from step 0.
  // 5. Remove all transitions that have become self transitions
  
  bool has_changed=false;
  list<NonEmittingNode*> node_loop;    // Will store the nodes making up a loop 
  
  do {
    has_changed=false;
    for (list<NonEmittingNode*>::const_iterator nenp=link_nodes.begin(); nenp!=link_nodes.end(); ++nenp) {
      node_loop.resize(0);
      if ((*nenp)->recursiveNENLoopSearch(node_loop)) {
	has_changed=mergeNENs(node_loop);
	break;
      }
    }
  } while (has_changed);

  for_each(link_nodes.begin(), link_nodes.end(), mem_fun(&NonEmittingNode::removeSelfTransitions));

}


// Bypass and remove any redundant non emitting nodes
void Decoder::simplifyNENs() {

  bool potential_redundant_NEN = true;

  // Repeatedly run through all the nodes until a complete pass is made in which no nodes are bypassed
  while (potential_redundant_NEN) {
    potential_redundant_NEN=false;
    for (list<NonEmittingNode*>::const_iterator nenp=link_nodes.begin(); nenp!=link_nodes.end(); ++nenp) {
      // Start and end nodes should never be removed
      if ((*nenp)==network_start_node) continue; 
      if ((*nenp)==network_end_node) continue;
      if ((*nenp)->bypassIfRedundant()) {
	delete *nenp;
	link_nodes.erase(remove(link_nodes.begin(), link_nodes.end(), *nenp), link_nodes.end());
	potential_redundant_NEN=true;
	break;
      }
    }
  }

}

bool Decoder::mergeNENs(list<NonEmittingNode*> &node_loop) {
  // Merge NEN in list into a single node (this is used for collapsing NEN loops)

  if (node_loop.size()<2) return false;

  // Make a new node representing all the nodes in the node loop

  NonEmittingNode *new_node=new NonEmittingNode(node_loop);

  if (find(node_loop.begin(), node_loop.end(), network_start_node)!=node_loop.end()) {
    setStartNode(new_node);
  }

  if (find(node_loop.begin(), node_loop.end(), network_end_node)!=node_loop.end()) {
    setEndNode(new_node);
  }

  // Remove loop nodes from NEN list and delete them
  list<NonEmittingNode*>::iterator end_pointer=link_nodes.end();
  for (list<NonEmittingNode*>::iterator nenp=node_loop.begin(), nenp_end=node_loop.end(); nenp!=nenp_end; ++nenp) 
    end_pointer=remove(link_nodes.begin(), end_pointer, *nenp);
  link_nodes.erase(end_pointer, link_nodes.end());

  sequence_delete(node_loop.begin(), node_loop.end());

  link_nodes.push_back(new_node);
  
  return true;
}


void Decoder::orderNENs() {
  // Place NENs in an order in which data may be passed through them consistently
  // i.e. if A connects to B connects to C then they must pull in the order A, B, C

  // i.e. must make sure that each NEN comes earlier in the list than any NEN that
  // has a PullTransition pointing to it. So:
  
  // 1. Clear flags on all NEN in the NEN list
  // 2. Flag all NENs pointed to by From Transitions of NENs in the NEN list
  // 3. Place unflagged NENs onto end of new list while removing from NEN list
  // 4. Repeat 1,2,3 until NEN list is empty
  // 5. Reverse copy the new NEN lst to the empty original NEN list

  list<NonEmittingNode*> new_list;    // Place to build list in correct order
  new_list.resize(link_nodes.size());
  list<NonEmittingNode*>::iterator new_copy_end=new_list.begin();
  
  while (link_nodes.size()!=0) {
    // Clear marks on all NENs in the NEN list
    for_each(link_nodes.begin(), link_nodes.end(), mem_fun(&NonEmittingNode::clearFlag));

    // Flag all NENs pointed to by From Transitions of NENs in the NEN list
    for_each(link_nodes.begin(), link_nodes.end(), mem_fun(&NonEmittingNode::flagFromNodes));

    // Push unflagged NENs onto back of new list
    new_copy_end=remove_copy_if(link_nodes.begin(), link_nodes.end(), new_copy_end, mem_fun(&NonEmittingNode::flagged));

    // Remove unflagged NENs from old list
    list<NonEmittingNode*>::iterator new_end=remove_if(link_nodes.begin(), link_nodes.end(), boost::compose_f_gx(logical_not<bool>(), (mem_fun(&NonEmittingNode::flagged))) );

    link_nodes.erase(new_end, link_nodes.end());
  }
  
  // We must now reverse new_list as remove_copy_if has added nodes to the end, not the beginning
  link_nodes.resize(new_list.size());
  reverse_copy(new_list.begin(), new_list.end(), link_nodes.begin());
  new_list.clear();
  
#if defined (_DEBUG)
  cerr << "NonEmittingNode list after ordering:\n";
  for (list<NonEmittingNode*>::iterator nenp=link_nodes.begin(), nenp_end=link_nodes.end(); nenp!=nenp_end; ++nenp) 
    cerr << **nenp;
  cerr << "\n";
#endif
  
}

void Decoder::cleanUp() {
  WordRecord::destroyAllWordRecords();   
}

NodePair Decoder::addNodePair(bool is_optional, bool is_repeatable) {
  NonEmittingNode *start_node=addNEN();
  NonEmittingNode *end_node=addNEN();

  if (is_optional) end_node->connectFrom(start_node, 1.0);
  if (is_repeatable) start_node->connectFrom(end_node, 1.0);

  return (NodePair(start_node, end_node));
}


RecoHypothesis *Decoder::get_compliant_hyp(regex_t *filter) {

  vector<RecoHypothesis*> *hypp = new vector<RecoHypothesis*>;
  int nbest=getNBestListSize();

  CTKStatus back_trace_status=CTK_FAILURE;
  
  try {
    // Set the size of the NBest list to the length by the filter
    setNBestListSize(FILTER_NBEST_N_CONST);
    back_trace_status=backTrace(*hypp, -NBEST_THRESHOLD_STEP_SIZE);
    // Reset the NBest list size for normal operation
    setNBestListSize(nbest);
  } catch (CTKError &error) {
    setNBestListSize(nbest);
    cerr << "Problems encountered constructing N best list." << endl;
  }

  Integer nhyp=hypp->size();
  
  Integer pos;
  for (pos=0; pos<nhyp; ++pos) {
    if (!(*hypp)[pos]->matches_filter(filter)) break;
  }

  // If can't find any valid hyp then reverst to the 1st best
  if (pos==nhyp) {
    pos=0;
  }
  
  RecoHypothesis* best_hyp;

  if (back_trace_status==CTK_SUCCESS)
    best_hyp= (*hypp)[pos];
  else
    best_hyp=NULL;
  
  delete hypp;

  return best_hyp;
}

//
//
//


// A new potential group has started - spawn a complimentary set of hypotheses
void Decoder::spawn_group_hypotheses(const Group &new_group) {

  // For each group starting the decoder list doubles in length - we keep the original decoder list
  // (which ignores the new group) and we add a copy in which the new group is added to each
  // decoder. i.e. if the decoders are [1, 2], [1, 3] and group 4 starts then we end up with
  // [1, 2], [1, 3], [1, 2, 4], [1, 3, 4]
  
  // Copy the hypotheses

  unsigned int N=group_hypothesis.size();
  group_hypothesis.reserve(N*2);
  copy(group_hypothesis.begin(), group_hypothesis.end(), back_inserter(group_hypothesis));

  renumber_group_hypotheses(group_hypothesis);
  
  // In one set the new group is assumed to be background
  for (vector<GroupHypothesis>::iterator ghp=group_hypothesis.begin(), ghp_end=ghp+N; ghp!=ghp_end; ++ghp)
    ghp->add_group_background(new_group); 

  // In the other set the new group is assumed to be speech
  for (vector<GroupHypothesis>::iterator ghp=group_hypothesis.begin()+N, ghp_end=group_hypothesis.end(); ghp!=ghp_end; ++ghp)
    ghp->add_group_speech(new_group); 
  
  // (Note - what has been done to the group_hypothesis vector mimics what the next function does to the
  // vectors of tokens that are in each HMM state) 
  
  // Signal to all nodes in the network that a new group has started causing the tokens to double up
  for_each(emitting_nodes.begin(), emitting_nodes.end(), mem_fun(&Node::newGroup));
  for_each(link_nodes.begin(), link_nodes.end(), mem_fun(&Node::newGroup));

  // Don't forget the start node! -it isn't included in link_nodes
  network_start_node->newGroup();

}


// Merge hypotheses which have the same masked group names
void Decoder::merge_group_hypotheses(const Group &dead_group, float merging_parameter) {
  // Perform the hypothesis score missing data vs present data normalising adjustment
  if (merging_parameter!=1.0) {
    float present_weight = dead_group.weight() * log(merging_parameter);
    float missing_weight = (dead_group.area()-dead_group.weight()) * log(merging_parameter);
    //    cerr << "MERGING: present_weight = " << present_weight << ", missing_weight = " << missing_weight << "\n"; 
    for (vector<GroupHypothesis>::iterator ghp=group_hypothesis.begin(), ghp_end=group_hypothesis.end(); ghp!=ghp_end; ++ghp) {
    int gh_slot = ghp->get_slot();
    bool has_group=ghp->has_group(dead_group.number());
    float adjustment = (has_group?present_weight:missing_weight);
    // cerr << "ADJUSTMENT: " << adjustment << "\n";
    for(list<EmittingNodeBase*>::iterator np=emitting_nodes.begin(), np_end=emitting_nodes.end(); np!=np_end; ++np)
      (*np)->adjustTokenScore(gh_slot, adjustment);
    for(list<NonEmittingNode*>::iterator np=link_nodes.begin(), np_end=link_nodes.end(); np!=np_end; ++np)
      (*np)->adjustTokenScore(gh_slot, adjustment);
    }
  }
  
  // Mask the state of the dead group in the group name
  for (vector<GroupHypothesis>::iterator ghp=group_hypothesis.begin(), ghp_end=group_hypothesis.end(); ghp!=ghp_end; ++ghp)
    ghp->make_masked_name(dead_group);
  
  // Sort the decoders according to their masked names 
  sort(group_hypothesis.begin(), group_hypothesis.end(), comp_group_hypotheses);

  vector<GroupHypothesis> new_list;
  vector<int> slot_list;
  vector<vector<int> > slot_listlist;
  
  GroupHypothesis hyp=group_hypothesis.back();
  group_hypothesis.pop_back();
  new_list.push_back(hyp);
  slot_list.push_back(hyp.get_slot());

  // i) Construct slot_listlist which is a list of lists of slots to be merged
  // e.g  [1,5], [2,6], [3,7], [4,8]  would mean merge slots 1 and 5, merge slots 2 and 6 etc
  // ii) Merge group hypotheses to form new (shorter) list
  while (group_hypothesis.size()>0) {
    GroupHypothesis hyp2=group_hypothesis.back();
    group_hypothesis.pop_back();
    //    cerr << hyp.get_masked_name() << " --- " << hyp2.get_masked_name() << "\n";
    if (hyp.get_masked_name()!=hyp2.get_masked_name()) {
      slot_listlist.push_back(slot_list);
      slot_list.clear();
      hyp=hyp2;
      new_list.push_back(hyp);
    }
    slot_list.push_back(hyp2.get_slot());
  }
  slot_listlist.push_back(slot_list);

  
  group_hypothesis=new_list;

  renumber_group_hypotheses(group_hypothesis);

  // Merge tokens from separate slots of each network state according to the merging pattern dictated by slot_listlist
  for(list<EmittingNodeBase*>::iterator np=emitting_nodes.begin(), np_end=emitting_nodes.end(); np!=np_end; ++np)
    (*np)->mergeTokens(slot_listlist);
  for(list<NonEmittingNode*>::iterator np=link_nodes.begin(), np_end=link_nodes.end(); np!=np_end; ++np)
    (*np)->mergeTokens(slot_listlist);
  // Don't forget the start node! -it isn't included in link_nodes
  network_start_node->mergeTokens(slot_listlist);
  
}



// Make mask according to the current group hypothesis under consideration
// Mapping:  active groups -> inmask value , 0 and inactive groups -> 0
// (Trying this for multisource with discrete masks)
void  Decoder::make_discrete_mask_hypothesis(const vector<Integer> &ingroup_mask, const vector<Float> &inmask, vector<Float> &outmask, bool inverted) {
  current_group_hypothesis->make_discrete_mask(ingroup_mask, inmask, outmask, inverted);
}

// Make mask according to the current group hypothesis under consideration
// Mapping: 0 and active groups -> inmask value;  inactive groups -> 1-inmask value
// (Trying this for multisource with soft masks)

void  Decoder::make_soft_mask_hypothesis(const vector<Integer> &ingroup_mask, const vector<Float> &inmask, vector<Float> &outmask, bool inverted) {
  current_group_hypothesis->make_soft_mask(ingroup_mask, inmask, outmask, inverted);
}

void Decoder::reset_current_group_hypothesis() {
  current_group_hypothesis=group_hypothesis.begin();
  Node::setTokenSlot(current_group_hypothesis->get_slot());
}

bool Decoder::next_group_hypothesis() {
  ++current_group_hypothesis;

  if (current_group_hypothesis==group_hypothesis.end()) {
    reset_current_group_hypothesis();
    return false;
  } else {
    Node::setTokenSlot(current_group_hypothesis->get_slot());
    return true;
  }
}


void  Decoder::renumber_group_hypotheses(vector<GroupHypothesis> &gh) {
  int slot_no=0;
  for (vector<GroupHypothesis>::iterator ghp=gh.begin(), ghp_end=gh.end(); ghp!=ghp_end; ++ghp, ++slot_no)
    ghp->set_slot(slot_no);

}



void Decoder::buildPronunciationTree(PronunciationTree &root, const vector<string> &some_hmms) {
  // Arrange word pronunciation into a tree structure branching from root.
  for(vector<string>::const_iterator hmmpp=some_hmms.begin(); hmmpp!=some_hmms.end(); ++hmmpp) {
    list<string> pronunciation = hmms->lookup_pronunciation(*hmmpp);
    string label=hmms->lookup_label(*hmmpp);
    root.addSequenceToTree(pronunciation, label);
  }
}

void Decoder::addPronunciationTreeBetween(const PronunciationTree &tree, NonEmittingNode *start_node, NonEmittingNode *end_node) {

  int pops=0;

  tree.resetTree();
  tree.parseTree(pops=0);  // steps into root node
  
  const PronunciationTree *ptp;

  list<NonEmittingNode *> attachment_point;

  attachment_point.push_back(start_node);
  
  while ((ptp=tree.parseTree(pops=0))!=NULL) {
    while (pops>0) {
      attachment_point.pop_back();
      --pops;
    }

    NonEmittingNode *new_node=NULL;
    string label = ptp->getLabel();
    if (label.size()!=0)
      new_node=end_node;

    //   cerr << *ptp << "\n";
    new_node=addBaseHMMByName(ptp->getNodeName(), attachment_point.back(), new_node, label);
    
    attachment_point.push_back(new_node);
    
  }
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: PronunciationTree                	         	              */
/*									      */
/*****************************************************************************/

void PronunciationTree::addSequenceToTree(const list<string> &sequence, const string &label) {

  PronunciationTree *new_node;
  PronunciationTree *current_node=this;
  list<string>::const_iterator sp, sp_end=sequence.end();
  for(sp=sequence.begin(); sp!=sp_end; ++sp) {
    new_node=&current_node->descendPronunciationTree(*sp);
    if (new_node==current_node) {
      break; // sequence is not in tree
    }
    current_node=new_node;
  }

  if (sp==sp_end) return; // sequence was already in tree;

  for(; sp!=sp_end; ++sp) {
    current_node=&current_node->growPronunciationTree(*sp);    
  }

  current_node->label=label;

}


PronunciationTree &PronunciationTree::descendPronunciationTree(const string &name) {
  //list<PronunciationTree>::iterator treep = find_if(children.begin(), children.end(), bind2nd(mem_fun_ref(&PronunciationTree::isMyNodeName), name));
  
  list<PronunciationTree>::iterator treep, treep_end=children.end();
  for (treep=children.begin(); treep!=treep_end; ++treep) {
    if (treep->isMyNodeName(name)) break;
  }
  
  if (treep==children.end()) 
    return *this;
  else
    return *treep;
}

PronunciationTree &PronunciationTree::growPronunciationTree(const string &name) {
  children.push_back(PronunciationTree(name));

  PronunciationTree &tree = children.back();

  return tree;
  
}

void PronunciationTree::resetTree() const {
  visited=false;
  current_child=children.begin();
  for_each(children.begin(), children.end(), mem_fun_ref(&PronunciationTree::resetTree));
}

const PronunciationTree *PronunciationTree::parseTree(int &popped) const {
  if (visited==false) {
    visited=true;
    return this;
  }

  while (current_child!=children.end()) {
    const PronunciationTree *ptp = current_child->parseTree(popped);
    if (ptp!=NULL) return ptp;
    current_child++;
  }

  popped+=1;
  
  return NULL;
  
}


ostream& operator<< (ostream& out, const PronunciationTree& tree) {
  out << "Tree: name = " << tree.node_name;
  if (tree.label.size()!=0) {
    out << " ( " << tree.label << " ) ";
  }
  return out;
}

////////////////////////////////////////

inline Boolean compare_hypotheses(RecoHypothesis *hyp1, RecoHypothesis *hyp2 ){
  return hyp1->score > hyp2->score;
}



/* End of ctk_decoder.cpp */
