/******************************************************************************/
/*									      */
/*	ctk_dsp.cpp	    		      			              */
/*									      */
/*	CTK Digitial Signal Processing Library                                */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"
 
#include <cmath>
 
#include "ctk_local.hh"
#include "ctk_error.hh"
#include "ctk_dsp.hh"

const char *StringArrays::VALID_WINDOW_SHAPES[] = {"RECTANGLE","HAMMING","HANNING","TRIANGLE","\0"};
const char *StringArrays::VALID_SCALES[] = {"ERB","MEL","LINEAR","\0"};

////////////////////////////////////////////////////////////////////////////////////////////////////////

void make_window(string shape, CTKVector &window, Integer win_size) {

  if (shape==string("RECTANGLE")) 
    make_rectangle_window(window, win_size);
  else if (shape==string("HANNING"))
    make_hanning_window(window, win_size);
  else if (shape==string("HAMMING"))
    make_hamming_window(window, win_size);
  else if (shape==string("TRIANGLE"))
    make_triangle_window(window, win_size);
  else {
    cerr << "ctk_spectro: Unknown window type :" << shape << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

}

void make_rectangle_window(CTKVector &window, Integer win_size) {
  window.resize(win_size, 1.0);
}

void make_hanning_window(CTKVector &window, Integer win_size) {
  window.resize(win_size);
  for (Integer i=0;i<win_size;++i)
    window[i]= 0.5*(1-cos(i*6.2831853/win_size));
}

void make_hamming_window(CTKVector &window, Integer win_size) {
  window.resize(win_size);
  for (Integer i=0;i<win_size;++i)
    window[i]= 0.54-0.46*cos(i*6.2831853/win_size);
}

void make_triangle_window(CTKVector &window, Integer win_size)
{
   window.resize(win_size);
   Integer half_size= (int)(win_size/2+0.5);
   for (Integer i=0, j=win_size-1; i<half_size; ++i, --j) 
     window[i] = window[j] = (Float)i/half_size;
   
}


void four1(CTKVector &data, Integer nn, Integer isign) {

  //  Replaces data[0..2*nn-1] by its discrete Fourier transform, if isign is input as 1; or replaces data[0..2*nn-1] by nn times its inverse discrete Fourier transform, if isign is input as data is a complex array of length nn or, equivalently, a real array of length 2*nn. nn MUST be an integer power of 2 (this is not checked for!).

  Integer n,mmax,m,j,istep,i;
  Float wtemp,wr,wpr,wpi,wi,theta;
  Float tempr,tempi;
  Float dj, djm1;
  
  n=nn << 1; j=1;
  for (i=1;i<n;i+=2) {
    // This is the bit-reversal section of the routine.
    if (j > i) {
      SWAP(data[j-1],data[i-1]); // Exchange the two complex numbers.
      SWAP(data[j],data[i]);
    }
    m=n >> 1;
    while (m >= 2 && j > m) {
      j -= m;
      m >>= 1;
    }
    j += m;
  }
  
  // Here begins the Danielson-Lanczos section of the routine.
  mmax=2;
  while (n > mmax) {
    // Outer loop executed log 2 nn times.
    istep=mmax << 1;
    theta=isign*(6.28318530717959/mmax);
    // Initialize the trigonometric recurrence.
    wtemp=sin(0.5*theta);
    wpr = -2.0*wtemp*wtemp;
    wpi=sin(theta);
    wr=1.0; wi=0.0;
    for (m=1;m<mmax;m+=2) {
      // Here are the two nested inner loops.
      for (i=m;i<=n;i+=istep) {
	j=i+mmax;
	// This is the Danielson-Lanczos formula:
	djm1=data[j-1]; dj=data[j];
	tempr=wr*djm1-wi*dj;
	tempi=wr*dj+wi*djm1;
	data[j-1]=data[i-1]-tempr;
	data[j]=data[i]-tempi;
	data[i-1] += tempr;
	data[i] += tempi;
      }
      wr=(wtemp=wr)*wpr-wi*wpi+wr; //Trigonometric recurrence.
      wi=wi*wpr+wtemp*wpi+wi;
    } mmax=istep;
  }
}




void realft(CTKVector &data, Integer isign) {
  //Calculates the Fourier transform of a set of n real-valued data points. Replaces this data (which is stored in array data[0..n-1]) by the positive frequency half of its complex Fourier transform. The real-valued  rst and last components of the complex transform are returned as elements data[0] and data[1], respectively. n must be a power of 2. This routine also calculates the inverse transform of a complex data array if it is the transform of real data. (Result in this case must be multiplied by 2/n.)
 Integer n=data.size();
  
 Integer i,i1,i2,i3,i4,np3;
 Float c1=0.5,c2,h1r,h1i,h2r,h2i;
 Float wr,wi,wpr,wpi,wtemp,theta; // Double precision for the trigonomet- ric recurrences.
 theta=3.141592653589793/(Float) (n>>1); // Initialize the recurrence.

 if (isign == 1) {
   c2 = -0.5;
   four1(data,n>>1,1); // The forward transform is here.
 } else {
   c2=0.5; // Otherwise set up for an inverse trans- form.
   theta = -theta;
 }
 wtemp=sin(0.5*theta);
 wpr = -2.0*wtemp*wtemp;
 wpi=sin(theta);
 wr=1.0+wpr;
 wi=wpi;
 np3=n+3;
 for (i=1;i<(n>>2);++i) {
   //Case i=1 done separately below.
   i4=1+(i3=np3-2-(i2=1+(i1=i+i)));
   h1r=c1*(data[i1]+data[i3]);
   // The two separate transforms are sep- arated out of data
   h1i=c1*(data[i2]-data[i4]);
   h2r = -c2*(data[i2]+data[i4]);
   h2i=c2*(data[i1]-data[i3]);
   data[i1]=h1r+wr*h2r-wi*h2i;
   //Here they are recombined to form the true transform of the origi- nal real data.
   data[i2]=h1i+wr*h2i+wi*h2r;
   data[i3]=h1r-wr*h2r+wi*h2i;
   data[i4] = -h1i+wr*h2i+wi*h2r;
   wr=(wtemp=wr)*wpr-wi*wpi+wr; // The recurrence.
   wi=wi*wpr+wtemp*wpi+wi;
 }

 if (isign == 1) {
   data[0] = (h1r=data[0])+data[1];
   // Squeeze the  rst and last data to- gether to get them all within the original array.
   data[1] = h1r-data[1];
 } else {
   data[0]=c1*((h1r=data[0])+data[1]);
   data[1]=c1*(h1r-data[1]);
   four1(data,n>>1,-1);
   //This is the inverse transform for the case isign=-1.
 }
}

void realft(CTKVector data, CTKVector &real, CTKVector &imag) {
  //Calculates the Fourier transform of a set of n real-valued data points. Replaces this data (which is stored in array data[0..n-1]) by the positive frequency half of its complex Fourier transform. The real-valued  rst and last components of the complex transform are returned as elements data[0] and data[1], respectively. n must be a power of 2. This routine also calculates the inverse transform of a complex data array if it is the transform of real data. (Result in this case must be multiplied by 2/n.)
 Integer n=data.size();
  
 Integer i,i1,i2,i3,i4,np3, j1, j2;
 Float c1=0.5,c2,h1r,h1i,h2r,h2i;
 Float wr,wi,wpr,wpi,wtemp,theta; // Double precision for the trigonomet- ric recurrences.
 theta=3.141592653589793/(Float) (n>>1); // Initialize the recurrence.

 
 c2 = -0.5;
 four1(data,n>>1,1); 

 wtemp=sin(0.5*theta);
 wpr = -2.0*wtemp*wtemp;
 wpi=sin(theta);
 wr=1.0+wpr;
 wi=wpi;
 np3=n+3;
 for (i=1;i<(n>>2);++i) {
   //Case i=1 done separately below.
   i4=1+(i3=np3-2-(i2=1+(i1=i+i)));
   j1=i1>>1;
   j2=(i3>>1)-1;
   
   // The two separate transforms are sep- arated out of data
   h1r=c1*(data[i1]+data[i3]);
   h1i=c1*(data[i2]-data[i4]);
   h2r = -c2*(data[i2]+data[i4]);
   h2i=c2*(data[i1]-data[i3]);

   //Here they are recombined to form the true transform of the origi- nal real data.
   real[j1]=h1r+wr*h2r-wi*h2i;
   imag[j1]=h1i+wr*h2i+wi*h2r;
   real[j2]=h1r-wr*h2r+wi*h2i;
   imag[j2] = -h1i+wr*h2i+wi*h2r;
   
   // The recurrence.
   wr=(wtemp=wr)*wpr-wi*wpi+wr; 
   wi=wi*wpr+wtemp*wpi+wi;
 }

 real[0] = c1*((h1r=data[0])+data[1]);
 imag[0] = 0;
 
 real[(n/2)-1] = c1*(h1r-data[1]);
 imag[(n/2)-1] = 0;

}


/*
  ALGORITHM 712, COLLECTED ALGORITHMS FROM ACM.
  THIS WORK PUBLISHED IN TRANSACTIONS ON MATHEMATICAL SOFTWARE,
  VOL. 18, NO. 4, DECEMBER, 1992, PP. 434-435.
  The function returns a normally distributed pseudo-random number
  with a given mean and standard devaiation.  Calls are made to a
  function subprogram which must return independent random
  numbers uniform in the interval (0,1).
  The algorithm uses the ratio of uniforms method of A.J. Kinderman
  and J.F. Monahan augmented with quadratic bounding curves.
*/


double random_gaussian(double mean,double stddev)
{
   double  q,u,v,x,y;

   /*  
      Generate P = (u,v) uniform in rect. enclosing acceptance region 
      Make sure that any random numbers <= 0 are rejected, since
      gaussian() requires uniforms > 0, but RandomUniform() delivers >= 0.
   */
   do {
      u = (Float)rand()/RAND_MAX;
      v = (Float)rand()/RAND_MAX;
      if (u <= 0.0 || v <= 0.0) {
          u = 1.0;
          v = 1.0;
      }
      v = 1.7156 * (v - 0.5);

      /*  Evaluate the quadratic form */
      x = u - 0.449871;
      y = fabs(v) + 0.386595;
      q = x * x + y * (0.19600 * y - 0.25472 * x);

      /* Accept P if inside inner ellipse */
      if (q < 0.27597)
         break;

      /*  Reject P if outside outer ellipse, or outside acceptance region */
    } while ((q > 0.27846) || (v * v > -4.0 * log(u) * u * u));

    /*  Return ratio of P's coordinates as the normal deviate */
    return (mean + stddev * v / u);
}



/* End of ctk_dsp.cpp */

