/******************************************************************************/
/*									      */
/*	ctk_error.hh	    						      */
/*									      */
/*	Declarations and definitions of CTK exception classes.		      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007	         		      */
/*									      */
/******************************************************************************/

#ifndef CTK_ERROR_HH
#define CTK_ERROR_HH

#include "ctk_local.hh"

typedef const UInteger ErrorCode;

/******************************************************************************/
/*									      */
/*	CLASS NAME: ErrorMessage					      */
/*									      */
/******************************************************************************/

class ErrorMessage {
private:
  ErrorCode code; 
  const char *text;

  // Declare but don't define to block implicit constructors
  ErrorMessage& operator=(const ErrorMessage&);

public:
  ErrorMessage(ErrorCode n, const char *s);
  ErrorMessage(const ErrorMessage &x);

  virtual ~ErrorMessage();
  const char *get_text() const;
};

//
//  Predefined Error Codes
//

static ErrorCode CTK_GCE_ID      = 100;
static ErrorCode CTK_AF_ID       = 101;

static ErrorCode MATH_GME_ID     = 200;
static ErrorCode MATH_DBZ_ID     = 201;

static ErrorCode ARRAY_GAE_ID    = 300;
static ErrorCode ARRAY_IRE_ID    = 301;
static ErrorCode ARRAY_ADE_ID    = 302;
static ErrorCode ARRAY_NCA_ID    = 303;
static ErrorCode ARRAY_IAC_ID    = 304;

static ErrorCode BLOCK_GBE_ID    = 400;
static ErrorCode BLOCK_UOS_ID    = 401;
static ErrorCode BLOCK_UIS_ID    = 402;
static ErrorCode BLOCK_CFOS_ID   = 403;
static ErrorCode BLOCK_CFIS_ID   = 404;
static ErrorCode BLOCK_CCS_ID    = 405;
static ErrorCode BLOCK_LC_ID     = 406;
static ErrorCode BLOCK_CFP_ID    = 407;
static ErrorCode BLOCK_CRP_ID    = 408;
static ErrorCode BLOCK_PHUB_ID   = 409;
static ErrorCode BLOCK_PTF_ID    = 410;
static ErrorCode BLOCK_PTI_ID    = 411;
static ErrorCode BLOCK_PTB_ID    = 412;
static ErrorCode BLOCK_CFSB_ID   = 413;
static ErrorCode BLOCK_DRE_ID    = 414;
static ErrorCode BLOCK_NUN_ID    = 415;
static ErrorCode BLOCK_POR_ID    = 416;

static ErrorCode FILE_GFE_ID     = 500;
static ErrorCode FILE_IFOF_ID    = 501;
static ErrorCode FILE_OFOF_ID    = 502;
static ErrorCode FILE_IFH_ID     = 503;
static ErrorCode FILE_CFCF_ID    = 504;

static ErrorCode PARAM_GPE_ID    = 600;
static ErrorCode PARAM_PNS_ID    = 601;

static ErrorCode SHELL_GSE_ID    = 700;
static ErrorCode SHELL_CFB_ID    = 701;

static ErrorCode WINDOW_BLOCK_ID = 1000;

//
//  Default English Error Messages
//

static ErrorMessage CTK_GCE_MESSAGE(CTK_GCE_ID,"(GCE) General CTK Error: %s %s %s\n\r");
static ErrorMessage CTK_AF_MESSAGE(CTK_AF_ID,"(AF) CTK Assertion Failure\n\r");

//

static ErrorMessage MATH_GME_MESSAGE(MATH_GME_ID,"(GME) General Math Error: %s\n\r");

static ErrorMessage MATH_DBZ_MESSAGE(MATH_DBZ_ID,"(DBZ) Divizion By Zero\n\r");

//

static ErrorMessage ARRAY_GAE_MESSAGE(ARRAY_GAE_ID,"(GAE) General Array Error: %s\n\r");
 
static ErrorMessage ARRAY_IRE_MESSAGE(ARRAY_IRE_ID,"(IRE) Index Range Error: %s is not in range [0,%s]\n\r");
static ErrorMessage ARRAY_ADE_MESSAGE(ARRAY_ADE_ID,"(ADE) Array Dimension Error: %s indices supplied for matrix of rank %s\n\r");

static ErrorMessage ARRAY_NCA_MESSAGE(ARRAY_NCA_ID,"(NCA) Non-Compatible Arrays: Operation= %s\n\r");
static ErrorMessage ARRAY_IAC_MESSAGE(ARRAY_IAC_ID,"(IAC) Invalid Array Construction\n\r");

//

static ErrorMessage BLOCK_GBE_MESSAGE(BLOCK_GBE_ID,"(GBE) General Block Error: %s\n\r");

static ErrorMessage BLOCK_UOS_MESSAGE(BLOCK_UOS_ID,"(UOS) Unconnected Output Socket: %s\n\r");
static ErrorMessage BLOCK_UIS_MESSAGE(BLOCK_UIS_ID,"(UIS) Unconnected Input Socket: %s\n\r");
static ErrorMessage BLOCK_CFOS_MESSAGE(BLOCK_CFOS_ID,"(CFOS) Cannot Find Output Socket: %s\n\r");
static ErrorMessage BLOCK_CFIS_MESSAGE(BLOCK_CFIS_ID,"(CFIS) Cannot Find Input Socket: %s\n\r");
static ErrorMessage BLOCK_CCS_MESSAGE(BLOCK_CCS_ID,"(CCS) Cannot Connect Sockets: (%s->%s)\n\r");

static ErrorMessage BLOCK_LC_MESSAGE(BLOCK_LC_ID,"(LC) Block diagram contains lose connections.\n\r");

static ErrorMessage BLOCK_CFP_MESSAGE(BLOCK_CFP_ID,"(CFP) Cannot Find Parameter: %s\n\r");
static ErrorMessage BLOCK_CRP_MESSAGE(BLOCK_CRP_ID,"(CRP) Cannot Register Parameter: %s\n\r");

static ErrorMessage BLOCK_PHUB_MESSAGE(BLOCK_PHUB_ID,"(PHUB) Parameter Has Unset Block Pointer: %s\n\r");
static ErrorMessage BLOCK_PTF_MESSAGE(BLOCK_PTF_ID,"(PTF) Parameter Type Is Float: %s = %s not valid \n\r");
static ErrorMessage BLOCK_PTI_MESSAGE(BLOCK_PTI_ID,"(PTI) Parameter Type Is Integer: %s = %s not valid\n\r");
static ErrorMessage BLOCK_PTB_MESSAGE(BLOCK_PTB_ID,"(PTB) Parameter Type Is Boolean: %s = %s not valid\n\r");
static ErrorMessage BLOCK_CFSB_MESSAGE(BLOCK_CFSB_ID,"(CFSB) Cannot find subblock: %s\n\r");
static ErrorMessage BLOCK_DRE_MESSAGE(BLOCK_DRE_ID,"(DRE) Network Dependency Resolution Error\n\r");
static ErrorMessage BLOCK_POR_MESSAGE(BLOCK_POR_ID,"(POR) Parameter Our of Range: %s = %s not valid\n\r");

static ErrorMessage BLOCK_NUN_MESSAGE(BLOCK_NUN_ID,"(NUN) Non unique block name: %s\n\r");

static ErrorMessage WINDOW_BLOCK_SHIFT_MESSAGE(WINDOW_BLOCK_ID,"(SHIFT) Window shift is greater than window size\n\r");

//

static ErrorMessage FILE_GFE_MESSAGE(FILE_GFE_ID,"(GFE) General FILE Error: %s\n\r");

static ErrorMessage FILE_IFOF_MESSAGE(FILE_IFOF_ID,"(IFOF) Input FILE Open Failure: %s\n\r");
static ErrorMessage FILE_OFOF_MESSAGE(FILE_OFOF_ID,"(OFOF) Output FILE Open Failure: %s\n\r");
static ErrorMessage FILE_IFH_MESSAGE(FILE_IFH_ID,"(IFH) Invalid File Header: %s\n\r");
static ErrorMessage FILE_CFCF_MESSAGE(FILE_CFCF_ID,"(CFCF) CTK FILE Open Failure: %s\n\r");

//

static ErrorMessage PARAM_GPE_MESSAGE(PARAM_GPE_ID,"(GPE) General Parameter Error: %s\n\r");

static ErrorMessage PARAM_PNS_MESSAGE(PARAM_PNS_ID,"(PNS) Parameter Not Set: %s\n\r");

//

static ErrorMessage SHELL_GSE_MESSAGE(PARAM_GPE_ID,"(GPE) General Shell Error: %s\n\r");

static ErrorMessage SHELL_CFB_MESSAGE(PARAM_PNS_ID,"(CFB) Cannot Find Block: %s\n\r");


///////////////////////////////////////////////////////////////////////////////////////////////////
//
//                 EXCEPTION CLASS HIERARCHY
//
//////////////////////////////////////////////////////////////////////////////////////////////////


/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKError - CTK Exception				      */
/*									      */
/******************************************************************************/



class CTKError {

private:
  // Declared but not defined
  CTKError &operator=(const CTKError &);
  
protected:
  ErrorMessage message;
  const Integer lineno;
  const char *filenm;
  const char *param1;
  const char *param2;
  const char *param3;


public:
  
  CTKError(const char *fn, Integer line, ErrorMessage msg=CTK_GCE_MESSAGE, const char *p1="\0", const char *p2="\0", const char *p3="\0");
  
  CTKError(const CTKError &an_error);
    
  virtual ~CTKError();

  virtual void debug_print(const char *caught_at_filenm=0, const Integer caught_at_lineno=0) const;
  virtual void print(const char *caught_at_filenm=0, const Integer caught_at_lineno=0) const;
  virtual void get_message(char *message_buffer) const;
    
protected:

  virtual void caught_print(const char *caught_filenm, const Integer caught_lineno) const;
    
};


class CTKErrorGCE: public CTKError {
public:
  CTKErrorGCE(const char *fn, Integer line, const char *p1="\0", const char *p2="\0", const char *p3="\0"):CTKError(fn,line,CTK_GCE_MESSAGE,p1,p2,p3){};
  virtual ~CTKErrorGCE(){};
};


/******************************************************************************/
/*									      */
/*	   CLASS NAME: CTKErrorAF - CTK Assertion Failure		      */
/*									      */
/******************************************************************************/


class CTKErrorAF: public CTKError {
public:
  CTKErrorAF(const char *fn, Integer line, ErrorMessage msg=CTK_AF_MESSAGE, const char *p1="\0", const char *p2="\0", const char *p3="\0"):CTKError(fn,line,msg,p1,p2,p3){};
  virtual ~CTKErrorAF(){};
};

/******************************************************************************/
/*									      */
/*	CLASS NAME: MathError - Math Error Exception Heirarchy		      */
/*									      */
/******************************************************************************/

class MathError: public CTKError {
public:
  MathError(const char *fn, const Integer line, ErrorMessage msg=MATH_GME_MESSAGE, const char *p1="\0", const char *p2="\0", const char *p3="\0"):CTKError(fn,line,msg,p1,p2,p3){};
  virtual ~MathError(){};
};

class MathErrorDBZ: public MathError {
public:
  MathErrorDBZ(const char *fn, const Integer line):MathError(fn,line,MATH_DBZ_MESSAGE){};
  virtual ~MathErrorDBZ(){};
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: ArrayError - Array Error Exception Heirarchy	      */
/*									      */
/******************************************************************************/

class ArrayError: public CTKError {
public:
  ArrayError(const char *fn, const Integer line, ErrorMessage msg=ARRAY_GAE_MESSAGE, const char *p1="\0", const char *p2="\0", const char *p3="\0"):CTKError(fn,line,msg,p1,p2,p3){};
  virtual ~ArrayError(){};
};

class ArrayErrorIRE: public ArrayError {

  char buffer_[CTK_CHAR_BUFFER_SIZE];

public:

  ArrayErrorIRE(const char *fn, const Integer line, Integer max, Integer x);
  
  virtual ~ArrayErrorIRE(){};
};

class ArrayErrorADE: public ArrayError {

  char buffer_[CTK_CHAR_BUFFER_SIZE];

public:
  ArrayErrorADE(const char *fn, const Integer line, Integer rank, Integer x);

  virtual ~ArrayErrorADE(){};
};

class ArrayErrorNCA: public ArrayError {
public:
  ArrayErrorNCA(const char *fn, const Integer line, const char *operation):ArrayError(fn,line,ARRAY_NCA_MESSAGE, operation){};
  virtual ~ArrayErrorNCA(){};
};

class ArrayErrorIAC: public ArrayError {
public:
  ArrayErrorIAC(const char *fn, const Integer line):ArrayError(fn,line,ARRAY_IAC_MESSAGE){};
  virtual ~ArrayErrorIAC(){};
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: BlockError - Block Processing Exception Heirarchy	      */
/*									      */
/******************************************************************************/


class BlockError: public CTKError {
public:
  BlockError(const char *fn, const Integer line, ErrorMessage msg=BLOCK_GBE_MESSAGE, const char *p1="\0", const char *p2="\0", const char *p3="\0"):CTKError(fn,line,msg,p1,p2,p3){};
  virtual ~BlockError(){};
};

class BlockErrorUOS: public BlockError {
public:
  BlockErrorUOS(const char *fn, const Integer line, const char *socket_name):BlockError(fn, line, BLOCK_UOS_MESSAGE, socket_name) {};
  virtual ~BlockErrorUOS(){};
};

class BlockErrorUIS: public BlockError {
public:
  BlockErrorUIS(const char *fn, const Integer line, const char *socket_name):BlockError(fn, line, BLOCK_UIS_MESSAGE, socket_name) {};
  virtual ~BlockErrorUIS(){};
};

class BlockErrorCFOS: public BlockError {
public:
  BlockErrorCFOS(const char *fn, const Integer line, const char *socket_name):BlockError(fn, line, BLOCK_CFOS_MESSAGE, socket_name) {};
  virtual ~BlockErrorCFOS(){};
};

class BlockErrorCFIS: public BlockError {
public:
  BlockErrorCFIS(const char *fn, const Integer line, const char *socket_name):BlockError(fn, line, BLOCK_CFIS_MESSAGE, socket_name) {};
  virtual ~BlockErrorCFIS() {};
};

class BlockErrorCCS: public BlockError {
public:
  BlockErrorCCS(const char *fn, const Integer line, const char *s1_name, const char *s2_name):BlockError(fn, line, BLOCK_CCS_MESSAGE, s1_name, s2_name) {};
  virtual ~BlockErrorCCS(){};
};


class BlockErrorLC: public BlockError {
public:
  BlockErrorLC(const char *fn, const Integer line):BlockError(fn, line, BLOCK_LC_MESSAGE) {};
  virtual ~BlockErrorLC(){};
};

class BlockErrorCFP: public BlockError {
public:
  BlockErrorCFP(const char *fn, const Integer line, const char *param_name):BlockError(fn, line, BLOCK_CFP_MESSAGE, param_name) {};
  virtual ~BlockErrorCFP(){};
};

class BlockErrorCRP: public BlockError {
public:
  BlockErrorCRP(const char *fn, const Integer line, const char *param_name):BlockError(fn, line, BLOCK_CRP_MESSAGE, param_name) {};
  virtual ~BlockErrorCRP(){};
};

class BlockErrorPHUB: public BlockError {
public:
  BlockErrorPHUB(const char *fn, const Integer line, const char *param_name):BlockError(fn, line, BLOCK_PHUB_MESSAGE, param_name) {};
  virtual ~BlockErrorPHUB(){};
};

class BlockErrorPTF: public BlockError {
public:
  BlockErrorPTF(const char *fn, const Integer line, const char *param_name, const char *param_value):BlockError(fn, line, BLOCK_PTF_MESSAGE, param_name, param_value) {};
  virtual ~BlockErrorPTF(){};
};

class BlockErrorPTI: public BlockError {
public:
  BlockErrorPTI(const char *fn, Integer line, const char *param_name, const char *param_value):BlockError(fn, line, BLOCK_PTI_MESSAGE, param_name, param_value) {};
  virtual ~BlockErrorPTI(){};
};

class BlockErrorPTB: public BlockError {
public:
  BlockErrorPTB(const char *fn, Integer line, const char *param_name, const char *param_value):BlockError(fn, line, BLOCK_PTB_MESSAGE, param_name, param_value) {};
  virtual ~BlockErrorPTB(){};
};

class BlockErrorPOR: public BlockError {
public:
  BlockErrorPOR(const char *fn, Integer line, const char *param_name, const char *param_value):BlockError(fn, line, BLOCK_POR_MESSAGE, param_name, param_value) {};
  virtual ~BlockErrorPOR(){};
};

class BlockErrorCFSB: public BlockError {
public:
  BlockErrorCFSB(const char *fn, Integer line, const char *subblock_name):BlockError(fn, line, BLOCK_CFSB_MESSAGE, subblock_name) {};
  virtual ~BlockErrorCFSB(){};
};

class BlockErrorDRE: public BlockError {
public:
  BlockErrorDRE(const char *fn, Integer line):BlockError(fn, line, BLOCK_DRE_MESSAGE) {};
  virtual ~BlockErrorDRE(){};
};

class BlockErrorNUN: public BlockError {
public:
  BlockErrorNUN(const char *fn, Integer line, const char *block_name):BlockError(fn, line, BLOCK_NUN_MESSAGE, block_name) {};
  virtual ~BlockErrorNUN(){};
};

/******************************************************************************/
/*									      */
/*	Derived Block Processing Exceptions				      */
/*									      */
/******************************************************************************/


class WindowBlockShiftError: public BlockError {
public:
  WindowBlockShiftError(const char *fn, const Integer line):BlockError(fn,line,WINDOW_BLOCK_SHIFT_MESSAGE){};
  virtual ~WindowBlockShiftError(){};
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: FileError - File Processing Exceptions		      */
/*									      */
/******************************************************************************/

class FileError: public CTKError {
public:
  FileError(const char *fn, const Integer line, ErrorMessage msg=FILE_GFE_MESSAGE, const char *p1="\0", const char *p2="\0", const char *p3="\0"):CTKError(fn,line,msg,p1,p2,p3){};
  virtual ~FileError(){};
};

class FileErrorGFE: public CTKError {
public:
  FileErrorGFE(const char *fn, const Integer line,  const char *p1="\0", const char *p2="\0", const char *p3="\0"):CTKError(fn,line,FILE_GFE_MESSAGE,p1,p2,p3){};
  virtual ~FileErrorGFE(){};
};


class FileErrorCFCF: public FileError {
public:
  FileErrorCFCF(const char *fn, const Integer line, const char *file_name):FileError(fn, line, FILE_CFCF_MESSAGE, file_name) {};
  virtual ~FileErrorCFCF(){};
};

class FileErrorIFOF: public FileError {
public:
  FileErrorIFOF(const char *fn, const Integer line, const char *file_name):FileError(fn, line, FILE_IFOF_MESSAGE, file_name) {};
  virtual ~FileErrorIFOF(){};
};

class FileErrorOFOF: public FileError {
public:
  FileErrorOFOF(const char *fn, const Integer line, const char *file_name):FileError
  (fn, line, FILE_OFOF_MESSAGE, file_name) {};
  virtual ~FileErrorOFOF(){};
};

class FileErrorIFH: public FileError {
public:
  FileErrorIFH(const char *fn, const Integer line, const char *file_name):FileError
  (fn, line, FILE_IFH_MESSAGE, file_name) {};
  virtual ~FileErrorIFH(){};
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamError - Parameter Processing Exceptions      	      */
/*									      */
/******************************************************************************/

class ParamError: public CTKError {
public:
  ParamError(const char *fn, const Integer line, ErrorMessage msg=PARAM_GPE_MESSAGE, const char *p1="\0", const char *p2="\0", const char *p3="\0"):CTKError(fn,line,msg,p1,p2,p3){};
  virtual ~ParamError(){};
};

class ParamErrorPNS: public ParamError {
public:
  ParamErrorPNS(const char *fn, const Integer line, const char *param_name):ParamError(fn, line, PARAM_PNS_MESSAGE, param_name) {};
  virtual ~ParamErrorPNS(){};
};



/******************************************************************************/
/*									      */
/*	CLASS NAME: ShellError - Errors detected by the CTK Shell     	      */
/*									      */
/******************************************************************************/

class ShellError: public CTKError {
public:
  ShellError(const char *fn, const Integer line, ErrorMessage msg=SHELL_GSE_MESSAGE, const char *p1="\0", const char *p2="\0", const char *p3="\0"):CTKError(fn,line,msg,p1,p2,p3){};
  virtual ~ShellError(){};
};

class ShellErrorCFB: public ShellError {
public:
  ShellErrorCFB(const char *fn, const Integer line, const char *block_name):ShellError(fn, line, SHELL_CFB_MESSAGE, block_name) {};
  virtual ~ShellErrorCFB(){};
};


#endif

/* End of ctk_error.hh */
