/******************************************************************************/
/*                                                                            */
/*      ctk_feature_vector.cpp                                                */
/*                                                                            */
/*      Support for feature vectors                                           */
/*                                                                            */
/*      Author: Jon Barker, Sheffield University                              */
/*                                                                            */
/*      CTK VERSION 1.3.5  Apr 22, 2007                              */
/*  */
/******************************************************************************/

#include "ctk-config.h"

#include <cmath>

#include <vector>
#include <algorithm>
#include <cassert>

#include "ctk_local.hh"
#include "ctk_error.hh"
#include "ctk_types.hh"

#include "ctk_feature_vector.hh"

/******************************************************************************/
/*									      */
/*	CLASS NAME: DeltaVector	                      	       	      */
/*									      */
/******************************************************************************/

DeltaVector::DeltaVector(){};

DeltaVector::DeltaVector(const CTKVector &frame, bool has_deltas): has_deltas_(has_deltas) {
  // Cast down from CTKVector (potentially double) to HMMFloat (normally float)

  non_delta_size_=frame.size();
  
  data_.resize(non_delta_size_);
  copy (frame.begin(), frame.end(), data_.begin());

  if (has_deltas_) non_delta_size_/=2;
  
  common_construction();
}
 
// Copy constructor
DeltaVector::DeltaVector(const DeltaVector &feature_vector):data_(feature_vector.data_), has_deltas_(feature_vector.has_deltas_), non_delta_size_(feature_vector.non_delta_size_) {

  common_construction();
}

// Assignment operator
DeltaVector &DeltaVector::operator=(const DeltaVector &delta_vector) {

  if (this==&delta_vector) return *this;

  data_=delta_vector.data_;
  has_deltas_=delta_vector.has_deltas_;
  non_delta_size_=delta_vector.non_delta_size_;
  
  return *this;
}

DeltaVector::~DeltaVector() {
}

void DeltaVector::common_construction() {
  
  data_begin_=data_.begin();
  data_end_with_deltas_=data_.end();
  data_end_without_deltas_=data_begin_+non_delta_size_;

}


const vector<HMMFloat>::const_iterator &DeltaVector::begin() const {
  return data_begin_;
}

const vector<HMMFloat>::const_iterator &DeltaVector::end(bool with_deltas) const {
  return with_deltas?data_end_with_deltas_:data_end_without_deltas_;
}

ostream& operator<< (ostream& out, const DeltaVector& x) {
  out << "STATIC FEATURES: ";
  for (vector<HMMFloat>::const_iterator fp=x.begin(), fp_end=x.end(false); fp!=fp_end; ++fp)
    out << *fp << " ";
  out << "\n";
  
  if (x.has_deltas()) {
    out << "DELTA FEATURES: ";
    for (vector<HMMFloat>::const_iterator fp=x.end(false), fp_end=x.end(true); fp!=fp_end; ++fp)
      out << *fp << " ";
    out << "\n";
  }

  return out;
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: FeatureVector	                      	       	      */
/*									      */
/******************************************************************************/

FeatureVector::FeatureVector(): DeltaVector(), upper_boundsp_(&data()), marginal_normalisation_(true) , marg_norm_(0){};

FeatureVector::FeatureVector(const CTKVector &frame, CTKVector *flower_bounds, CTKVector *fupper_bounds, bool has_deltas, Float md_weight/*=1.0*/): DeltaVector(frame, has_deltas), upper_boundsp_(&data()), md_weight_(md_weight), marginal_normalisation_(true), marg_norm_(0) {
  // Cast down from CTKVector (potentially double) to HMMFloat (normally float)
  
  if (fupper_bounds) {
#if defined (_DEBUG)
    assert(fupper_bounds->size()==frame.size());
#endif
    upper_bounds_.resize(fupper_bounds->size());
    copy(fupper_bounds->begin(), fupper_bounds->end(), upper_bounds_.begin());  
    upper_boundsp_=&upper_bounds_;
  }

  if (flower_bounds) {
    lower_bounds_.resize(flower_bounds->size());
    copy(flower_bounds->begin(), flower_bounds->end(), lower_bounds_.begin());
  } else {
    lower_bounds_.resize(0);
    lower_bounds_.resize(frame.size(), 0.0);  // lower bounds = 0.0 by default
  }
  
  common_construction();
}

// Copy constructor
FeatureVector::FeatureVector(const FeatureVector &feature_vector): DeltaVector(feature_vector), lower_bounds_(feature_vector.lower_bounds_), upper_bounds_(feature_vector.upper_bounds_), upper_boundsp_(&data()),  md_weight_(feature_vector.md_weight_), marginal_normalisation_(feature_vector.marginal_normalisation_), marg_norm_(0) {

  if (!upper_bounds_.size())
    upper_boundsp_=&upper_bounds_;
  
  common_construction();
}

// Assignment operator
FeatureVector &FeatureVector::operator=(const FeatureVector &feature_vector) {

  if (this==&feature_vector) return *this;

  delete marg_norm_;
  marg_norm_=0;

  DeltaVector::operator=(feature_vector);

  lower_bounds_=feature_vector.lower_bounds_;
  
  upper_bounds_=feature_vector.upper_bounds_;
  md_weight_=feature_vector.md_weight_;
  
  upper_boundsp_=&data();
  if (!upper_bounds_.size())
    upper_boundsp_=&upper_bounds_;
  
  common_construction();

  marginal_normalisation_=feature_vector.marginal_normalisation_;
  
  return *this;
}

FeatureVector::~FeatureVector() {
  delete marg_norm_;
}

void FeatureVector::common_construction() {
  lower_bounds_begin_=lower_bounds_.begin();
  lower_bounds_end_with_deltas_=lower_bounds_.end();
  lower_bounds_end_without_deltas_=lower_bounds_begin_+non_delta_size();
  
  upper_bounds_begin_=upper_boundsp_->begin();
  upper_bounds_end_with_deltas_=upper_boundsp_->end();
  upper_bounds_end_without_deltas_=upper_bounds_begin_+non_delta_size();
}


const vector<HMMFloat>::const_iterator &FeatureVector::lower_bounds_begin() const {
  return lower_bounds_begin_;
}

const vector<HMMFloat>::const_iterator &FeatureVector::lower_bounds_end(bool with_deltas) const {
  return with_deltas?lower_bounds_end_with_deltas_:lower_bounds_end_without_deltas_;
}

const vector<HMMFloat>::const_iterator &FeatureVector::upper_bounds_begin() const {
  return upper_bounds_begin_;
}

const vector<HMMFloat>::const_iterator &FeatureVector::upper_bounds_end(bool with_deltas) const {
  return with_deltas?upper_bounds_end_with_deltas_:upper_bounds_end_without_deltas_;
}

const vector<HMMFloat>::const_iterator &FeatureVector::marg_norm_begin() const {
  if (!marg_norm_) build_marg_norm();
  return marg_norm_begin_;
}

const vector<HMMFloat>::const_iterator &FeatureVector::marg_norm_end(bool with_deltas) const {
  if (!marg_norm_) build_marg_norm();
  return with_deltas?marg_norm_end_with_deltas_:marg_norm_end_without_deltas_;
}

void FeatureVector::set_marginal_normalisation(bool x) {
  marginal_normalisation_=x;
}

void FeatureVector::build_marg_norm() const {
  marg_norm_ = new vector<HMMFloat> (0);

  for (vector<HMMFloat>::const_iterator upper_boundsp=upper_bounds_begin_,lower_boundsp=lower_bounds_begin_,lower_bounds_end=lower_bounds_end_with_deltas_; lower_boundsp!=lower_bounds_end; ++upper_boundsp, ++lower_boundsp) {
    marg_norm_->push_back(marginal_normalisation_?(log(md_weight_/ (*upper_boundsp-*lower_boundsp) )) : 0); 
  }
  
  marg_norm_begin_=marg_norm_->begin();
  marg_norm_end_with_deltas_=marg_norm_->end();
  marg_norm_end_without_deltas_=marg_norm_begin_+non_delta_size();
}

ostream& operator<< (ostream& out, const FeatureVector& x) {
  
  out << "FeatureVector:\n";

  operator<<(out, (DeltaVector)x);

  out << "LOWER BOUNDS (STATIC): ";
  for (vector<HMMFloat>::const_iterator fp=x.lower_bounds_begin(), fp_end=x.lower_bounds_end(false); fp!=fp_end; ++fp)
    out << *fp << " ";
  out << "\n";
 
  if (x.has_deltas()) {
    out << "LOWER BOUNDS (DELTA): ";
    for (vector<HMMFloat>::const_iterator fp=x.lower_bounds_end(false), fp_end=x.lower_bounds_end(true); fp!=fp_end; ++fp)
      out << *fp << " ";
    out << "\n";
  }
  
  out << "UPPER BOUNDS (STATIC): ";
  for (vector<HMMFloat>::const_iterator fp=x.upper_bounds_begin(), fp_end=x.upper_bounds_end(false); fp!=fp_end; ++fp)
    out << *fp << " ";
  out << "\n";
 
  if (x.has_deltas()) {
    out << "UPPER BOUNDS (DELTA): ";
    for (vector<HMMFloat>::const_iterator fp=x.upper_bounds_end(false), fp_end=x.upper_bounds_end(true); fp!=fp_end; ++fp)
      out << *fp << " ";
    out << "\n";
  }

  out << "MD_WEIGHT = " << x.md_weight() << "\n";

  out << "MARGINAL NORMALISATION (YES/NO): " << x.marginal_normalisation_ << "\n";
  
  return out;
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: MaskVector	                         	       	      */
/*									      */
/******************************************************************************/

MaskVector::MaskVector(): log_mask_(0), log_one_minus_mask_(0) {};

MaskVector::MaskVector(const CTKVector &frame, bool has_deltas): DeltaVector(frame, has_deltas), log_mask_(0), log_one_minus_mask_(0) {
  // Cast down from CTKVector (potentially double) to HMMFloat (normally float)
  
  common_construction();
}

// Copy constructor
MaskVector::MaskVector(const MaskVector &mask_vector): DeltaVector(mask_vector), log_mask_(0), log_one_minus_mask_(0) {

  common_construction();
}

// Assignment operator
MaskVector &MaskVector::operator=(const MaskVector &mask_vector) {

  if (this==&mask_vector) return *this;

  DeltaVector::operator=(mask_vector);
  
  delete log_mask_;
  delete log_one_minus_mask_;
  log_mask_=log_one_minus_mask_=0;
    
  common_construction();

  return *this;
}

MaskVector::~MaskVector() {
  delete log_mask_;
  delete log_one_minus_mask_;
}

void MaskVector::common_construction() {
}

const vector<HMMFloat>::const_iterator &MaskVector::log_mask_begin() const {
  if (!log_mask_) build_log_mask();
  return log_mask_begin_;
}

const vector<HMMFloat>::const_iterator &MaskVector::log_one_minus_mask_begin() const {
  if (!log_one_minus_mask_) build_log_one_minus_mask();
  return log_one_minus_mask_begin_;
}


void MaskVector::build_log_mask() const {
  log_mask_ = new vector<HMMFloat> (0);
  for (vector<HMMFloat>::const_iterator maskp=begin(),mask_end=end(false); maskp!=mask_end; ++maskp)
  log_mask_->push_back(log(*maskp));
  
  log_mask_begin_=log_mask_->begin();
}


void MaskVector::build_log_one_minus_mask() const {
  log_one_minus_mask_ = new vector<HMMFloat> (0);
  for (vector<HMMFloat>::const_iterator maskp=begin(),mask_end=end(false); maskp!=mask_end; ++maskp) 
    log_one_minus_mask_->push_back(log(1.0-*maskp));
  log_one_minus_mask_begin_=log_one_minus_mask_->begin();
}

ostream& operator<< (ostream& out, const MaskVector& x) {

  out << "MaskVector:\n";

  operator<<(out, (DeltaVector)x);

  return out;

}

/* End of ctk_feature_vector.cpp */
