/******************************************************************************/
/*                                                                            */
/*      ctk_feature_vector.hh                                                           */
/*                                                                            */
/*      Support for HMMs                                                      */
/*                                                                            */
/*      Author: Jon Barker, Sheffield University                              */
/*                                                                            */
/*      CTK VERSION 1.3.5  Apr 22, 2007                              */
/*  */
/******************************************************************************/

#ifndef CTK_FEATURE_VECTOR_HH
#define CTK_FEATURE_VECTOR_HH

#include <vector>

#include "ctk_local.hh"
#include "ctk_error.hh"
#include "ctk_types.hh"

#include "ctk_HMM_types.hh"

/******************************************************************************/
/*									      */
/*	CLASS NAME: DeltaVector	                        	       	      */
/*									      */
/******************************************************************************/

class DeltaVector {

private:
  vector<HMMFloat> data_;
  
  bool has_deltas_;
  int non_delta_size_;

  vector<HMMFloat>::const_iterator data_begin_; 
  vector<HMMFloat>::const_iterator data_end_with_deltas_; 
  vector<HMMFloat>::const_iterator data_end_without_deltas_;
  
public:

  DeltaVector();
  DeltaVector(const CTKVector &frame, bool has_deltas);
  DeltaVector(const DeltaVector &delta_vector);

  DeltaVector &operator=(const DeltaVector &delta_vector);
  
  ~DeltaVector();
  
  bool has_deltas() const {return has_deltas_;}
  int non_delta_size() const {return non_delta_size_;}
  int size() const {return data_.size();}
  
  const vector<HMMFloat>::const_iterator &begin() const;
  const vector<HMMFloat>::const_iterator &end(bool with_deltas) const;

  friend ostream& operator<< (ostream& out, const class DeltaVector &x);

protected:

  vector<HMMFloat> &data() {return data_;}
  
private:
  
  void common_construction();

};

/******************************************************************************/
/*									      */
/*	CLASS NAME: FeatureVector	                      	       	      */
/*									      */
/******************************************************************************/

class FeatureVector: public DeltaVector  {

private:
  vector<HMMFloat> lower_bounds_;
  vector<HMMFloat> upper_bounds_;
  vector<HMMFloat> *upper_boundsp_;

  HMMFloat md_weight_;  // This is used as part of the marginal normalisation - an set scaling between present and missing data evidence.

  bool marginal_normalisation_;  // True if marginal normalisation is to be applied, else false.
  
  mutable vector<HMMFloat> *marg_norm_;

  vector<HMMFloat>::const_iterator lower_bounds_begin_;
  vector<HMMFloat>::const_iterator lower_bounds_end_with_deltas_;
  vector<HMMFloat>::const_iterator lower_bounds_end_without_deltas_;
  
  vector<HMMFloat>::const_iterator upper_bounds_begin_;
  vector<HMMFloat>::const_iterator upper_bounds_end_with_deltas_;
  vector<HMMFloat>::const_iterator upper_bounds_end_without_deltas_;
  
  mutable vector<HMMFloat>::const_iterator marg_norm_begin_;
  mutable vector<HMMFloat>::const_iterator marg_norm_end_with_deltas_;
  mutable vector<HMMFloat>::const_iterator marg_norm_end_without_deltas_;

public:

  FeatureVector();
  FeatureVector(const CTKVector &frame, CTKVector *flower_bounds, CTKVector *fupper_bounds, bool ahas_deltas, Float md_weight=1.0);
  FeatureVector(const FeatureVector &feature_vector);

  FeatureVector &operator=(const FeatureVector &feature_vector);
  
  ~FeatureVector();

  const vector<HMMFloat>::const_iterator &lower_bounds_begin() const;
  const vector<HMMFloat>::const_iterator &lower_bounds_end(bool with_deltas) const;

  const vector<HMMFloat>::const_iterator &upper_bounds_begin() const;
  const vector<HMMFloat>::const_iterator &upper_bounds_end(bool with_deltas) const;

  const vector<HMMFloat>::const_iterator &marg_norm_begin() const;
  const vector<HMMFloat>::const_iterator &marg_norm_end(bool with_deltas) const;

  HMMFloat md_weight() const {return md_weight_;}

  void set_marginal_normalisation(bool x);  // Enable or disable the marginal normalisation factor
  
  friend ostream& operator<< (ostream& out, const FeatureVector& x);

private:
  
  void common_construction();
  void build_marg_norm() const;
  
};

/******************************************************************************/
/*									      */
/*	CLASS NAME: MaskVector	                         	       	      */
/*									      */
/******************************************************************************/

class MaskVector: public DeltaVector  {

private:

  mutable vector<HMMFloat> *log_mask_;
  mutable vector<HMMFloat> *log_one_minus_mask_;
  
  mutable vector<HMMFloat>::const_iterator log_mask_begin_; 
  mutable vector<HMMFloat>::const_iterator log_one_minus_mask_begin_; 

public:

  MaskVector();
  MaskVector(const CTKVector &frame, bool ahas_deltas);
  MaskVector(const MaskVector &mask_vector);

  MaskVector &operator=(const MaskVector &mask_vector);
  
  ~MaskVector();

  const vector<HMMFloat>::const_iterator &log_mask_begin() const;
  const vector<HMMFloat>::const_iterator &log_one_minus_mask_begin() const;

  friend ostream& operator<< (ostream& out, const MaskVector& x);

private:
  
  void common_construction();
  void build_log_mask() const;
  void build_log_one_minus_mask() const;

};

#endif

/* End of ctk_feature_vector.hh */
