/******************************************************************************/
/*									      */
/*	ctk_group.cpp	        	      		              */
/*									      */
/*	A spectro-temporal region - aka a group, or a fragment                */
/*									      */
/*	Author: Jon Barker, Sheffield University            		      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007     	         	      */
/*									      */
/******************************************************************************/

#include "ctk_group.hh"  

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: Group                                                    */
/*                                                                            */
/******************************************************************************/

Group::Group():number_(0), label_(0), start_frame_(0), end_frame_(0), cog_frame_acc_(0.0), min_channel_(0), max_channel_(0), cog_channel_acc_(0.0), area_(0), weight_(0.0), source_(0) {};

Group::Group(int number, int label):number_(number), label_(label), start_frame_(0), end_frame_(0), cog_frame_acc_(0.0), min_channel_(0), max_channel_(0), cog_channel_acc_(0.0), area_(0), weight_(0.0), source_(0) {};

void Group::add_point(int frame_number, int channel_number, float weight/*=1.0*/) {
  if (area_==0) {
    start_frame_=frame_number;
    end_frame_=frame_number;
    max_channel_=channel_number;
    min_channel_=channel_number;
  } else {
    if (channel_number>max_channel_) max_channel_=channel_number;
    if (channel_number<min_channel_) min_channel_=channel_number;
    if (frame_number>end_frame_) end_frame_=frame_number;
    if (frame_number<start_frame_) start_frame_=frame_number; 
  }
  
  cog_channel_acc_+=(weight*channel_number);
  cog_frame_acc_+=(weight*frame_number);
  
  ++area_;
  weight_+=weight;
}


void Group::remove_point(int frame_number, int channel_number, float weight/*=1*/) {
  if (area_==0)
    return;
  
  cog_channel_acc_-=(weight*channel_number);
  cog_frame_acc_-=(weight*frame_number);
  
  --area_;
  weight_-=weight;
  
}


void Group::output(FILE *a_file) const {
  fprintf(a_file,"%d %d %d %d %f %d %d %f %d %f %d\n", number(), label(), start_frame(), end_frame(), cog_frame(), min_channel(), max_channel(), cog_channel(), area(), weight(), source());
}

CTKStatus Group::input(FILE *a_file) {
  float cog_frame;
  float cog_channel;
  
  if (fscanf(a_file,"%d %d %d %d %f %d %d %f %d %f %d\n",&number_, &label_, &start_frame_, &end_frame_, &cog_frame, &min_channel_, &max_channel_, &cog_channel, &area_, &weight_, &source_)!=10)
    return CTK_FAILURE;

  cog_channel_acc_=cog_channel*weight_;
  cog_frame_acc_=cog_frame*weight_;

  return CTK_SUCCESS;
}

//Group information 
ostream& operator<< (ostream& out, const Group& x) {
  cerr << "Group Number = " << x.number_ << "\n";
  cerr << "Group Label = " << x.label_ << "\n";
  cerr << " Group Start Frame = " << x.start_frame_ << "; End Frame = " << x.end_frame_ << "\n";
  cerr << " Group Min Channel = " << x.min_channel_ << "; Max Channel = " << x.max_channel_ << "\n";
  cerr << " Group C of G: Frame = " << x.cog_frame() << "; Channel = " << x.cog_channel() << "\n";
  cerr << " Group Area = " << x.area_ << "\n";
  cerr << " Group Weight = " << x.weight_ << "\n";
  cerr << " Group Source = " << x.source_ << "\n";
  return out;
}
/* End of ctk_group.cpp */
