/******************************************************************************/
/*									      */
/*	ctk_group.hh   			             		      */
/*									      */
/*	Class declarations for ctk_group.cpp	               	      */
/*									      */
/*	Author: Jon Barker, Sheffield University              	      	      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007                   		      */
/*									      */
/******************************************************************************/

 
#ifndef CTK_GROUP_HH
#define CTK_GROUP_HH

#include "ctk_local.hh"
#include <cstdio>
#include <deque>
#include <map>


/******************************************************************************/
/*									      */
/*	CLASS NAME: Group 		              	       	      */
/*									      */
/******************************************************************************/

// Note some care is needed when interpreting the max and min channel and frame. They 
// are the max and min of any point that has ever belonged to the group. If points are
// removed from the group the max and min values will not be effected.

class Group {

 private:

  int number_;        // Group id 
  int label_;        // Group input label
  int start_frame_;  // index of frame where group first appears
  int end_frame_;    // index of frame where group last appears
  float cog_frame_acc_;  // accumulator for centre of gravity framewise
  int min_channel_;  // index of channel of lowest frequency extent
  int max_channel_;  // index of channel of highest frequency extent
  float cog_channel_acc_; // accumulator for centre of gravity channelwise
  int area_;         // total number of spectro temporal pixels
  float weight_;     // weight of group
  int source_;      // id of the source to which the group belongs
  
 public:
  
  Group();

  Group(int number, int label);

  void add_point(int frame_number, int channel_number, float weight=1.0);

  // Note that if remove point is used, then the max and min value may become 'out of date'
  void remove_point(int frame_number, int channel_number, float weight=1.0);
  
  int number() const {return number_;}
  int label() const {return label_;}
  int start_frame() const {return start_frame_;}
  int end_frame() const {return end_frame_;}
  float cog_frame() const {return (weight_==0.0)?0.0:cog_frame_acc_/weight_;}
  int min_channel() const {return min_channel_;}
  int max_channel() const {return max_channel_;}
  float cog_channel() const {return (weight_==0.0)?0.0:cog_channel_acc_/weight_;}
  int area() const {return area_;}
  float weight() const {return weight_;}
  int source() const {return source_;}
  
  void output(FILE *a_file) const;
  CTKStatus input(FILE *a_file);
  
  friend ostream& operator<< (ostream& out, const Group& x);
};
  
#endif


/* End of ctk_group.hh */
