/****************************************************************************/
/*									      */
/*	ctk_lattice.cpp	    						      */
/*									      */
/*	Implemenation for Lattice	- lattices a HTK format grammar file	      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <list>
#include <map>
#include <string>
#include <fstream>
#include <sstream>

#include <functional>
#include <algorithm>

#include "ctk_local.hh"
#include "ctk_error.hh"
 
#include "ctk_lattice.hh"

#include "ctk_function_classes.hh"

#define CTK_CHAR_BUFFER_SIZE 1024

/******************************************************************************/
/*									      */
/*	CLASS NAME: LatticeArc	         				      */
/*									      */
/******************************************************************************/

ostream& operator<< (ostream& out, const LatticeArc& arc) {

   cerr << "J: ID = " << arc.id_ << "; START= " << arc.from_ << "; END= " << arc.to_ << "\n";
 return out;
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: LatticeNode	         				      */
/*									      */
/******************************************************************************/

ostream& operator<< (ostream& out, const LatticeNode& node) {

  cerr << "I: ID = " << node.id_ << "; W= " << node.name_ << "\n";
  return out;
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: Lattice	         				      */
/*									      */
/******************************************************************************/

Lattice::Lattice(std::istream &ifstr){
  
  char buffer[CTK_CHAR_BUFFER_SIZE];

  while (ifstr.getline(buffer,CTK_CHAR_BUFFER_SIZE,'\n')) {
    interpret_line(string(buffer));
  }

  //  cerr << *this << "\n";
  
}

void Lattice::interpret_line(const string &line) {

  std::istringstream ifstr(line.c_str());

  string lhs, rhs;

  if (is_comment(ifstr)) return;
  
  parse_assignment(ifstr, lhs, rhs);

  if (lhs=="VERSION") interpret_VERSION(ifstr, rhs);
  else if (lhs=="N") interpret_N(ifstr, rhs);
  else if (lhs=="I") interpret_I(ifstr, rhs);
  else if (lhs=="J") interpret_J(ifstr, rhs);
  
}

bool Lattice::is_comment(std::istringstream &ifstr) {

  char x;
  
  ifstr>>x;
  if (x=='#') return true;

  ifstr.putback(x);
  
  return false;
  
}

bool Lattice::parse_assignment(std::istringstream &ifstr, string &lhs, string &rhs) {

  char x;

  while (ifstr>>x && x!='=') lhs+=x;
  while (ifstr.get(x) && isspace(x));

  if (!ifstr) return false;
  
  ifstr.putback(x);
  
  while (ifstr.get(x) && !isspace(x)) rhs+=x;

  return (lhs.size()!=0 && rhs.size()!=0);
}

Lattice::~Lattice(){

};

bool Lattice::interpret_VERSION(std::istringstream &, const string &){
  return true;
}

bool Lattice::interpret_N(std::istringstream &, const string &){
  return true;
}

bool Lattice::interpret_I(std::istringstream &ifstr, const string &id_string){
  string name;
  int id=string_to_int(id_string);

  string lhs, rhs;
  while (parse_assignment(ifstr, lhs, rhs)) {
    if (lhs=="W") name=rhs;
    lhs.erase(); rhs.erase();
  }

  nodes[id]=LatticeNode(id, name);

  return true;
}

bool Lattice::interpret_J(std::istringstream &ifstr, const string &id_string){

  int start=0, end=0;

  int id=string_to_int(id_string);

  string lhs, rhs;
  while (parse_assignment(ifstr, lhs, rhs)) {
    if (lhs=="S") start=string_to_int(rhs);
    else if (lhs=="E") end=string_to_int(rhs);
    lhs.erase(); rhs.erase();
  }

  arcs[id]=LatticeArc(id, start, end);
		   
  return true;
}


int Lattice::string_to_int(const string &s) {
  int x;
  std::istringstream str(s.c_str());
  str>>x;
  return x;
}

ostream& operator<< (ostream& out, const Lattice& lattice) {

  out << "Nodes:\n";
  for (map<int, LatticeNode>::const_iterator nodep=lattice.nodes.begin(); nodep!=lattice.nodes.end(); ++nodep) 
    out << nodep->second;    
  out << "Arc:\n";
  for (map<int, LatticeArc>::const_iterator arcp=lattice.arcs.begin(); arcp!=lattice.arcs.end(); ++arcp) 
    out << arcp->second;    
  return out;
}


// Construct a network to represent this lattice - return true on success else false
// The network is constructed between a given pair of NENs belonging to 'network'

CTKStatus Lattice::buildNetwork(Decoder &network, NetworkNodePair network_node_pair) {

  // Convert lattice object into a full decoder network.

  map<int, NonEmittingNode*> start_node_map;
  map<int, NonEmittingNode*> end_node_map;

  network_node_pair.start->setID(-1);
  network_node_pair.end->setID(-2);
  
  for (map<int, LatticeNode>::const_iterator lnmp=nodes.begin(), lnmp_end=nodes.end(); lnmp!=lnmp_end; ++lnmp) {
    const LatticeNode *lnp=&lnmp->second;
    if (network.getHMMByName(lnp->name())==NULL) {
      // Lattice node is not an HMM therefore it represents a Null node (an NEN)
      //      cerr << "Adding Null Node:  " << lnp->name() << "\n";
      NonEmittingNode *node = network.addNEN();
      node->setID(lnp->id());
      start_node_map[lnp->id()]=node;
      end_node_map[lnp->id()]=node;
    } else {
      // Lattice node represents an HMM, add the HMM to the decoder network
      //      cerr << "Adding HMM: " << lnp->name() << "\n";
      NodePair node_pair=network.addHMMByName(lnp->name());
      start_node_map[lnp->id()]=node_pair.start;
      end_node_map[lnp->id()]=node_pair.end;
      node_pair.start->setID(lnp->id());
      node_pair.end->setID(lnp->id());
    }
  }
  
  for (map<int, LatticeArc>::const_iterator lamp=arcs.begin(), lamp_end=arcs.end(); lamp!=lamp_end; ++lamp) {
    const LatticeArc *lap=&lamp->second;
    NonEmittingNode *from_node=end_node_map[lap->from()];
    NonEmittingNode *to_node=start_node_map[lap->to()];
    if (from_node==NULL || to_node==NULL) {
      cerr << "invalid arc:\n" << *lap << "\n";
    }
    nodes[lap->from()].unsetEndNode();
    nodes[lap->to()].unsetStartNode();
    to_node->connectFrom(from_node, 1.0);
  }

  
  // Connect this lattice network between the supplied network node pair
  int start_node_id, end_node_id;
  CTKStatus status=find_start_and_end_nodes(start_node_id, end_node_id);
  if (status==CTK_SUCCESS) {
    //    cerr << "Start node = " << start_node_id << "; end node = " << end_node_id << "\n";
    
    network_node_pair.end->connectFrom(end_node_map[end_node_id], 1.0);
    start_node_map[start_node_id]->connectFrom(network_node_pair.start, 1.0);
  } else {
    cerr << "Lattice does not have unique start and end nodes.\n Error in SLF grammar file?";
    
  }
  
  return status;
}

CTKStatus Lattice::find_start_and_end_nodes(int &start_node_id, int &end_node_id) const {

  int n_starts=0, n_ends=0;
  
  for (map<int, LatticeNode>::const_iterator lnmp=nodes.begin(), lnmp_end=nodes.end(); lnmp!=lnmp_end; ++lnmp) {
    const LatticeNode *lnp=&lnmp->second;
    if (lnp->start_node()) {
      start_node_id=lnp->id();
      n_starts++;
    }
    if (lnp->end_node()) {
      end_node_id=lnp->id();
      n_ends++;
    }
  }

  // A valid lattice should have only one start node and one end node.
  if (n_starts==1 && n_ends==1)
    return CTK_SUCCESS;
  else
    return CTK_FAILURE;
  
}

/* End of ctk_lattice.cpp */
