/******************************************************************************/
/*									      */
/*	ctk_param.cpp	    						      */
/*									      */
/*	Implementation for Param class hierarchy			      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		         	      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <cstdio>

#include <cmath>
#include <algorithm>
#include <sstream>

#include "ctk_local.hh"
#include "ctk_error.hh"
#include "ctk_function_classes.hh"

#include "ctk_block.hh"
#include "ctk_param.hh"

extern string expand_environment_variables(const string &word);

/******************************************************************************/
/*									      */
/*	Non CLASS functions						      */
/*									      */
/******************************************************************************/

string bool_string(bool x) {
  return (x)?string("true"):string("false");
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamList						      */
/*									      */
/******************************************************************************/

ParamList::ParamList(CTKObject *this_objectp): params(0), objectp(this_objectp), override_flag(false) {
}

ParamList::~ParamList() {
  sequence_delete(params.begin(), params.end());
}


Boolean ParamList::register_parameter(Param *paramp, UInteger parameter_flags/*=0*/) {

  if (objectp==NULL) {cerr << "Internal error!" << endl; exit(-1);}
  
  paramp->set_objectp(objectp); // must set this first so that get_param_pathname() operates correctly

  // check parameter name is unique
  for (ParamIt paramp2=params.begin(); paramp2!=params.end(); ++paramp2) {
    if ((*paramp2)->is_my_basepathname(paramp->get_param_pathname())) {
      cerr << "*** " << paramp->get_param_basepathname() << endl;
      cerr << "*** " << (*paramp2)->get_param_basepathname() << endl;
      paramp->set_objectp(NULL);
      return false;
    }
  }

  params.push_back(paramp);

  if (parameter_flags&CTK_PARAM_TYPE_CONST) paramp->set_const();
  if (parameter_flags&CTK_PARAM_TYPE_HIDDEN) paramp->set_hidden();
  if (parameter_flags&CTK_PARAM_TYPE_TRANSMITTED) paramp->set_transmitted();
  if (parameter_flags&CTK_PARAM_TYPE_DEPRECATED) paramp->set_deprecated();
  
  return true;
}


void ParamList::display(ostream &outfile, const string &this_name) const {
  ParamConstIt paramp=find_if(params.begin(), params.end(), bind2nd(mem_fun(&Param::is_my_fullname),this_name));
  if (paramp!=params.end()) (*paramp)->display(outfile, "\0");
}

void ParamList::display_all(ostream &outfile, const string &space) const {

  for (ParamConstIt pp = params.begin(); pp != params.end(); ++pp) 
    (*pp)->display(outfile, space);
  
}

void ParamList::fprintf_XML_all(FILE *fp) const {

  for (ParamConstIt pp = params.begin(); pp != params.end(); ++pp) 
    (*pp)->fprintf_XML(fp);
  
}


CTKStatus ParamList::set(const string &this_name, const string &value) {

  ParamIt paramp=find_if(params.begin(), params.end(), bind2nd(mem_fun(&Param::is_my_pathname),this_name));
  if (paramp!=params.end()) {
    (*paramp)->set_with_string(value, override_flag);
    return CTK_SUCCESS;
  } else
    return CTK_FAILURE;
}

CTKStatus ParamList::set_unresolved_value(const string &this_name, const string &value) {

  ParamIt paramp=find_if(params.begin(), params.end(), bind2nd(mem_fun(&Param::is_my_pathname),this_name));
  if (paramp!=params.end()) {
    (*paramp)->set_unresolved_value(value);
    return CTK_SUCCESS;
  } else
    return CTK_FAILURE;
}


CTKStatus ParamList::set(const string &this_name, Float value) {

  ParamIt paramp=find_if(params.begin(), params.end(), bind2nd(mem_fun(&Param::is_my_pathname),this_name));
  if (paramp!=params.end()) {
    (*paramp)->set_with_float(value, override_flag);
     return CTK_SUCCESS;
  } else
    return CTK_FAILURE;
}

CTKStatus ParamList::unset(const string &this_name) {

  ParamIt paramp=find_if(params.begin(), params.end(), bind2nd(mem_fun(&Param::is_my_pathname),this_name));
  if (paramp!=params.end()) {
    (*paramp)->unset();
     return CTK_SUCCESS;
  } else
    return CTK_FAILURE;
}
   
CTKStatus ParamList::set_const(const string &this_name) {
 
  ParamIt paramp=find_if(params.begin(), params.end(), bind2nd(mem_fun(&Param::is_my_pathname),this_name));
  if (paramp!=params.end()) {
    (*paramp)->set_const();
    return CTK_SUCCESS;
  } else
    return CTK_FAILURE;
}

CTKStatus ParamList::set_hidden(const string &this_name) {
 
  ParamIt paramp=find_if(params.begin(), params.end(), bind2nd(mem_fun(&Param::is_my_pathname),this_name));
  if (paramp!=params.end()) {
    (*paramp)->set_hidden();
    return CTK_SUCCESS;
  } else
    return CTK_FAILURE;
}
 
CTKStatus ParamList::set_transmitted(const string &this_name) {
 
  ParamIt paramp=find_if(params.begin(), params.end(), bind2nd(mem_fun(&Param::is_my_pathname),this_name));
  if (paramp!=params.end()) {
    (*paramp)->set_transmitted();
    return CTK_SUCCESS;
  } else
    return CTK_FAILURE;
}
 
CTKStatus ParamList::unset_const(const string &this_name) {
 
  ParamIt paramp=find_if(params.begin(), params.end(), bind2nd(mem_fun(&Param::is_my_pathname),this_name));
  if (paramp!=params.end()) {
    (*paramp)->unset_const();
    return CTK_SUCCESS;
  } else
    return CTK_FAILURE;
}

CTKStatus ParamList::unset_hidden(const string &this_name) {
 
  ParamIt paramp=find_if(params.begin(), params.end(), bind2nd(mem_fun(&Param::is_my_pathname),this_name));
  if (paramp!=params.end()) {
    (*paramp)->unset_hidden();
    return CTK_SUCCESS;
  } else
    return CTK_FAILURE;
}

CTKStatus ParamList::unset_transmitted(const string &this_name) {
 
  ParamIt paramp=find_if(params.begin(), params.end(), bind2nd(mem_fun(&Param::is_my_pathname),this_name));
  if (paramp!=params.end()) {
    (*paramp)->unset_transmitted();
    return CTK_SUCCESS;
  } else
    return CTK_FAILURE;
}

CTKStatus ParamList::set_arg_number(const string &this_name, Integer arg_number) {

  ParamIt paramp=find_if(params.begin(), params.end(), bind2nd(mem_fun(&Param::is_my_pathname),this_name));
  if (paramp!=params.end()) {
    (*paramp)->set_arg_number(arg_number);
    return CTK_SUCCESS;
  } else
    return CTK_FAILURE;
}

Param* ParamList::make_subparameter(const string &this_name, const string &parent_param_name) {

  ParamIt paramp=find_if(params.begin(), params.end(), bind2nd(mem_fun(&Param::is_my_fullname),this_name));
  if (paramp!=params.end()) {
    return (*paramp)->return_parent_parameter(parent_param_name);
  } else
    return NULL;
}

void ParamList::clone(ParamList *cloned_param_list, const vector<Block*> b) const{
  
  string subparam_fullname;
  string parent_param_name;

  Param *parent_paramp;
  Boolean found;

  for (ParamConstIt paramp=params.begin(); paramp!=params.end(); ++paramp) {
    subparam_fullname=(*paramp)->get_subparam_fullname();
    parent_param_name=(*paramp)->get_param_name();
    if (subparam_fullname.size()==0) {
      // register parameter
      cloned_param_list->set(parent_param_name, (*paramp)->get_value_string());
    } else {
      found=false;
      for (BlockConstIt blockpp=b.begin(); blockpp!=b.end(); ++blockpp) {
	if ((parent_paramp=(*blockpp)->get_parameters()->make_subparameter(subparam_fullname, parent_param_name))!=NULL) {
	  found=true;
	  // register parameter
	  if (cloned_param_list->register_parameter(parent_paramp)==false)  throw(BlockErrorCRP(__FILE__, __LINE__,  parent_paramp->get_param_fullname().c_str()));
	  break;
	}
      }
      if (found==false) throw(BlockErrorCFP(__FILE__, __LINE__, subparam_fullname.c_str()));
    }
  }
}


void ParamList::copy_settings(const ParamList *list_to_copy) {

  for (ParamIt paramp = params.begin(); paramp!=params.end(); ++paramp) {
    for (ParamConstIt paramp_copy = list_to_copy->begin(); paramp_copy!=list_to_copy->end(); ++paramp_copy) {
      if ((*paramp)->is_my_name((*paramp_copy)->get_param_name())) {
	(*paramp)->set_from_param(*paramp_copy);
	break;
      }
    }
  }
}

void ParamList::clear_transmitted_parameters() {
  for (ParamIt paramp = params.begin(); paramp!=params.end(); ++paramp) {
    if (((*paramp)->get_transmitted_flag()==true))
      (*paramp)->unset();
  }
}

CTKStatus ParamList::copy_transmitted_settings(const ParamList *list_to_copy, ParamIt &dodgy_parameter) {

  // Fill transmitted parameters from the values of those with a matching name in list_to_copy
  
  for (ParamIt paramp = params.begin(); paramp!=params.end(); ++paramp) {
    if (((*paramp)->get_transmitted_flag()==true)) {
      for (ParamConstIt paramp_copy = list_to_copy->begin(); paramp_copy!=list_to_copy->end(); ++paramp_copy) {
	if ((*paramp)->is_my_name((*paramp_copy)->get_param_name())) {

	  if ((*paramp)->get_set_flag()==false) {
	    // If parameter is not set, then inherit value from feeding block's parameter
	    (*paramp)->set_value_from_param(*paramp_copy);
	    break;
	  } else {
	    // If already set then check the values are consistent. If they are inconsistent return FAILURE and set dodgy_parameter
	    if ((*paramp)->type()==PARAM_FLOAT) {
	      // Special case for float because "10.000" == "10.0" etc
	      ParamFloat* x1= dynamic_cast<ParamFloat*>(*paramp);
	      ParamFloat* x2= dynamic_cast<ParamFloat*>(*paramp_copy);
	      if (x1==NULL || x2==NULL) {cerr << "internal error\n"; exit(-1);}
	      if (x1->get_value()!=x2->get_value()) {
		cerr << x1->get_value() << " " << x2->get_value() << "\n";
		dodgy_parameter=paramp;
		return CTK_FAILURE;
	      }
	    } else
	      // For other types just compare their string value.
	      if (string((*paramp)->get_value_string())!=string((*paramp_copy)->get_value_string())) {
		dodgy_parameter=paramp;
		return CTK_FAILURE;
	      }
	  }
	  
	}
      }
    }
  }

  return CTK_SUCCESS;
}



void ParamList::unlock(){override_flag=true;};

void ParamList::relock(){override_flag=false;}


/******************************************************************************/
/*									      */
/*	CLASS NAME: Param						      */
/*									      */
/******************************************************************************/

/******************************************************************************/
/*									      */
/* Param - Public methods						      */
/*									      */
/******************************************************************************/


Param::Param(const string &this_name, Param *this_subparamp/*=NULL*/): subparamp(this_subparamp), objectp(NULL), name(this_name), helptext(), unit("\0"), fullname(), pathname(), unresolved_value(), arg_number(0), deprecated_flag(false), const_flag(false), transmitted_flag(false), hidden_flag(false), dirty_flag(false), validatorp_(NULL), value_string(), set_flag(false), baseparamp(NULL), dirty_string_flag(true) // Initialise to true (i.e. dirty) - it will be turned to 'false' (clean) when the string has been constructed for the first time

{

  helptext="No help available for this parameter\n";
  
  baseparamp=((subparamp==NULL)?this:subparamp->baseparamp);

}


Param::~Param() {
  delete validatorp_;
}

// Mark value string as out of date.
void Param::invalidate_value_string() {
  set_flag=true;
  dirty_flag=true;
  dirty_string_flag=true;
}

// Update value string and mark as up to date.
void Param::set_value_string(const string &s) const{
  value_string=s;
  dirty_string_flag=false;
}

Boolean Param::get_set_flag() const {
  return baseparamp->set_flag;
}

string Param::getname() const {
  return name;
}


void Param::set_helptext(string text) {helptext=text;}
void Param::set_unit(string some_unit) {unit=some_unit;} 

void Param::rename(string text) {name=text;}

/******************************************************************************/
/*									      */
/* Param - Protected methods						      */
/*									      */
/******************************************************************************/

//  PURE VIRTUAL: virtual string Param::get_value_string()=0;

Boolean Param::set_from_param(Param *a_param, bool set_properties/*=true*/) {
  set_arg_number(a_param->get_arg_number());
  set_unresolved_value(a_param->get_unresolved_value());
  
  if (a_param->get_set_flag()==true) {
    if (a_param->get_deprecated_flag()!=true)
      if (type()==PARAM_FLOAT)
	set_with_float(dynamic_cast<ParamFloat*>(a_param)->get_value(), true);
      else
	set_with_string(a_param->get_value_string(), true);
    
    if (a_param->validatorp() != NULL)
      validatorp_=new Validator(*(a_param->validatorp()));
    
    if (set_properties) {
      if (a_param->get_hidden_flag()==true)
	set_hidden(); else unset_hidden();
      if (a_param->get_const_flag()==true)
	set_const(); else unset_const();
      if (a_param->get_transmitted_flag()==true)
	set_transmitted(); else unset_transmitted();
      if (a_param->get_dirty_flag()==true)
	set_dirty(); else unset_dirty();
      if (a_param->get_deprecated_flag()==true)
	set_deprecated(); else unset_deprecated();
    }
  } else
    baseparamp->set_flag=false;
  
  return true;
}

Boolean  Param::set_with_string(const string &s, Boolean override_flag) {

  if (baseparamp!=this)
    return ((Param*)baseparamp)->set_with_string(s, override_flag);

  if (override_flag==false && (get_hidden_flag() || get_const_flag()))
    return false;

  if (deprecated_flag==true) {
    cerr << "WARNING: SCRIPT USES DEPRECATED PARAMETER ! (" <<  get_param_fullname() <<"). PARAMETER WILL BE IGNORED.\n";
    return false;
  }
    
  my_set_with_string(expand_environment_variables(s), override_flag);

  set_new_value_string(s);

  return true;
}

//  PURE VIRTUAL: virtual Boolean  Param::my_set_with_string(const string &s, Boolean override_flag)=0;

Boolean Param::set_with_float(Float, Boolean) {
  cerr<<"Internal Error. Not a numeric type" << endl; exit(-1);
  return false;
}

Boolean Param::get_deprecated_flag() const {
  return baseparamp->deprecated_flag;
}

Boolean Param::get_hidden_flag() const {
  return baseparamp->hidden_flag;
}

Boolean Param::get_const_flag() const {
  return baseparamp->const_flag;
}

Boolean Param::get_transmitted_flag() const {
  return baseparamp->transmitted_flag;
}

Boolean Param::get_dirty_flag() const {
  return baseparamp->dirty_flag;
}

void Param::set_new_value_string(const string &s) {
  set_flag=true;
  set_dirty_flag();
  set_value_string(s);
}

string Param::get_param_pathname() const{

    if (objectp==NULL) throw(BlockErrorPHUB(__FILE__, __LINE__, name.c_str()));

    pathname=objectp->prefix_object_fullname(name);

    return pathname;
}


void Param::clean_param_string(string &s) const {

   replace(s.begin(), s.end(), '{', ' '); 
   replace(s.begin(), s.end(), '}', ' '); 
   replace(s.begin(), s.end(), ',', ' '); 
}




/******************************************************************************/
/*									      */
/* Param - Private methods  (visible to fried class, ParamList 		      */
/*									      */
/******************************************************************************/

Boolean Param::is_my_name(string this_name) const {
  return name==this_name;
}

Boolean Param::is_my_fullname(string this_name) const {
  return get_param_fullname()==this_name;
}

Boolean Param::is_my_pathname(string this_name) const {
  return get_param_pathname()==this_name;
}

Boolean Param::is_my_basepathname(string this_name) const {
  return get_param_basepathname()==this_name;
}

Boolean Param::am_i_a_baseparam() const {return subparamp==NULL;}

Integer Param::get_arg_number() const {return  baseparamp->arg_number;}

bool Param::has_unresolved_value() const {return baseparamp->unresolved_value.size()!=0;}

string Param::get_unresolved_value() const {
  return baseparamp->unresolved_value;
}

void Param::display( ostream &outfile, const string &space, bool detailed/*=false*/) const {
  if (get_hidden_flag()==false || detailed) {
    if (get_set_flag())
      outfile << space << "PARAMETER: " << name << " = " << get_value_string();
    else
      outfile << space << "PARAMETER: " << name << " (unset) ";

    if (get_transmitted_flag())
      outfile <<" (transmitted) ";
    
    if (subparamp!=NULL) outfile << "(-->" << subparamp->get_param_fullname() << ")";
    if (get_const_flag())
      outfile << space << " [const] ";
    outfile << endl;
  }

  if (detailed) {
    // For detailed parameter output
    outfile << "HIDDEN = " << bool_string(get_hidden_flag()) << endl;
    outfile << "DEPRECATED = " << bool_string(get_deprecated_flag()) << endl;
    outfile << "DIRTY = " << bool_string(get_dirty_flag()) << endl;
    outfile << "BASEPARAM = " << bool_string(am_i_a_baseparam()) << endl;
  }

}

void Param::fprintf_XML( FILE *fp) const {
  fprintf(fp,"<%s> %s </%s>\n",name.c_str(),get_value_string().c_str(),name.c_str());
}

string Param::get_param_name() const {return name;}

string Param::get_param_fullname() const {
  if (objectp==NULL) throw(BlockErrorPHUB(__FILE__, __LINE__, (baseparamp->name).c_str()));

  fullname = objectp->prefix_object_name(name);
  return fullname;
};

string Param::get_param_basepathname() const {
  
  if (baseparamp!=this) {
    return baseparamp->get_param_pathname();
  } else {
    return get_param_pathname();
  }
}

string Param::get_subparam_fullname() const {
  if (subparamp==NULL) return string();
  return subparamp->get_param_fullname();
}


void Param::unset() {
  if (baseparamp==this) {
    set_flag=false;
  }
    else baseparamp->unset();
}

void Param::set_deprecated() {
  baseparamp->deprecated_flag=true;
}

void Param::set_hidden() {
  baseparamp->hidden_flag=true;
}

void Param::set_const() {
  baseparamp->const_flag=true;
}

void Param::set_transmitted() {
  baseparamp->transmitted_flag=true;
}

void Param::set_dirty() {
  baseparamp->dirty_flag=true;
}

void Param::unset_deprecated() {
  baseparamp->deprecated_flag=false;
}

void Param::unset_hidden() {
  baseparamp->hidden_flag=false;
}

void Param::unset_const() {
  baseparamp->const_flag=false;
}

void Param::unset_transmitted() {
  baseparamp->transmitted_flag=false;
}

void Param::unset_dirty() {
  baseparamp->dirty_flag=false;
}

void Param::set_unresolved_value(const string &value) {
  baseparamp->unresolved_value=value;
}

void Param::set_arg_number(Integer number) {
  baseparamp->arg_number=number;
}

CTKStatus Param::validate(int value) {
  if (validatorp_==NULL) return CTK_SUCCESS;
  return validatorp_->validate(value);
}

CTKStatus Param::validate(double value) {
  if (validatorp_==NULL) return CTK_SUCCESS;
  return validatorp_->validate(value);
}

CTKStatus Param::install_validator(Validator *validatorp) {
  if (validatorp_ == NULL) {
    validatorp_=validatorp;
    return CTK_SUCCESS;
  } else
    return CTK_FAILURE;
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamInt						      */
/*									      */
/******************************************************************************/


ParamInt::ParamInt(const string &this_name, Integer this_value,  Param *this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value(this_value) {
  set_flag=true;
}

ParamInt::ParamInt(const string &this_name,  Param *this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value(0) {
  set_flag=false;
}

string ParamInt::get_value_string() const {
  if ((ParamInt*)baseparamp!=this)
    return ((ParamInt*)baseparamp)->get_value_string();
  else {
    if (dirty_string_flag==false) return value_string; // Value string is up to date
    
    // Rebuild value string before returning it
    int x=(int)fabs((double)value);
    if (x<2) x=2;
    char *tmp_value_string=new Character[Integer(log10((double)x)+2.0+(value<0))];
    sprintf(tmp_value_string,"%d",value);
    value_string=tmp_value_string;
    delete[] tmp_value_string;

    dirty_string_flag=false;
    return value_string;
  }
}

Integer ParamInt::get_value() const {
  
  if ((ParamInt*)baseparamp!=this)
    return ((ParamInt*)baseparamp)->get_value();
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value;
  }
}

void ParamInt::my_set_with_string(const string &s, Boolean) {

  Integer newvalue;
  char crap[CTK_CHAR_BUFFER_SIZE+1];

  int nread=sscanf(s.c_str(), "%d%s", &newvalue, crap);
  
  if ((nread == 0) ||
      (nread>1 && (strlen(crap)>0)))
    throw(BlockErrorPTI(__FILE__, __LINE__, get_param_fullname().c_str(), s.c_str()));
    
  if (validate(newvalue)==CTK_FAILURE)
    throw(BlockErrorPOR(__FILE__, __LINE__, get_param_fullname().c_str(), s.c_str()));
  
  value=newvalue;
  
}

Boolean ParamInt::set_with_float(Float x, Boolean override_flag) {
  if (baseparamp!=this)
    return ((ParamInt*)baseparamp)->set_with_float(x, override_flag);
  else {
    if (override_flag==false && (get_hidden_flag() || get_const_flag())) return false;
    if (validate(x)==CTK_FAILURE) return false;
    value=(Integer)x;
  }

  invalidate_value_string(); // value string is no longer upto date
  return true;
}


Param* ParamInt::return_parent_parameter(const string &this_name) {
  if (set_flag)
    return new ParamInt(this_name, value, this);
  else
    return new ParamInt(this_name, this);
}



/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamFloat						      */
/*									      */
/******************************************************************************/


ParamFloat::ParamFloat(const string &this_name, Float this_value, Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value(this_value) {
  set_flag=true;
}

ParamFloat::ParamFloat(const string &this_name,  Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value(0.0) {
  set_flag=false;
}

Float ParamFloat::get_value() const {
  
  if ((ParamFloat*)baseparamp!=this)
    return ((ParamFloat*)baseparamp)->get_value();
  else {
 
    if (set_flag==false) {
      throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    }
    return value;
  }
}

void ParamFloat::my_set_with_string(const string &s, Boolean) {

  Float newvalue;
  char crap[CTK_CHAR_BUFFER_SIZE+1];

  int nread=sscanf(s.c_str(), "%lf%s", &newvalue, crap);

  if ((nread == 0) ||
      (nread>1 && (strlen(crap)>0)))
    throw(BlockErrorPTF(__FILE__, __LINE__, get_param_fullname().c_str(), s.c_str()));
  
  if (validate(newvalue)==CTK_FAILURE)
    throw(BlockErrorPOR(__FILE__, __LINE__, get_param_fullname().c_str(), s.c_str()));

  value=newvalue;
  
}
 
Boolean ParamFloat::set_with_float(Float x, Boolean override_flag) {
  if (baseparamp!=this)
    return ((ParamFloat*)baseparamp)->set_with_float(x, override_flag);
  else {
    if (override_flag==false && (get_hidden_flag() || get_const_flag())) return false;
    if (validate(x)==CTK_FAILURE) return false;
   value=x;
  }

  invalidate_value_string(); // value string is no longer upto date
  return true;
}

string ParamFloat::get_value_string() const {
  if ((ParamFloat*)baseparamp!=this)
    return ((ParamFloat*)baseparamp)->get_value_string();
  else {
    if (dirty_string_flag==false) return value_string; // Value string is up to date

    // Rebuild value string before returning it
    Float x=fabs((double)value);
    if (x<2.0) x=2.0;
    char *tmp_value_string= new Character[Integer(log10((double)x)+6+(value<0))]; // allow space for '-' sign.
    sprintf(tmp_value_string,"%5.3f",value);
    value_string=tmp_value_string;
    delete[] tmp_value_string;

    dirty_string_flag=false;
    return value_string;
  }
}


Param* ParamFloat::return_parent_parameter(const string &this_name) {
  if (set_flag)
    return new ParamFloat(this_name, value, this);
  else
    return new ParamFloat(this_name, this);
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamBool						      */
/*									      */
/******************************************************************************/


ParamBool::ParamBool(const string &this_name, Boolean this_value, Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value(this_value) {
  set_flag=true;
}

ParamBool::ParamBool(const string &this_name,  Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value(false) {
   set_flag=false;
}

Boolean ParamBool::get_value() const {
  
  if ((ParamBool*)baseparamp!=this)
    return ((ParamBool*)baseparamp)->get_value();
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value;
  }
}

void ParamBool::my_set_with_string(const string &s, Boolean) {
  
  char c=s[0];
  if (c == '1' || c == 't' || c == 'T' || c == 'y' || c == 'Y') 
    value=true;
  else if (c == '0' || c == 'f' || c == 'F' || c == 'n' || c == 'N') 
    value=false;
  else {
    throw(BlockErrorPTB(__FILE__, __LINE__, get_param_fullname().c_str(), s.c_str()));
  }

}

Boolean ParamBool::set_with_float(Float x, Boolean override_flag) {
  if (baseparamp!=this)
    return ((ParamBool*)baseparamp)->set_with_float(x, override_flag);
  else {
    if (override_flag==false && (get_hidden_flag() || get_const_flag())) return false;
    value=(x==0?false:true);
  }
  

  invalidate_value_string(); // value string is no longer upto date
  return true;
}

string ParamBool::get_value_string() const {
  if ((ParamBool*)baseparamp!=this)
    return ((ParamBool*)baseparamp)->get_value_string();
  else {
    if (dirty_string_flag==false) return value_string; // Value string is up to date

    // Rebuild value string before returning it
    value_string=(value?"Yes":"No");
    dirty_string_flag=false;
    return value_string;
  }
}

Param* ParamBool::return_parent_parameter(const string &this_name) {
  if (set_flag)
    return new ParamBool(this_name, value, this);
  else
    return new ParamBool(this_name, this);
}



/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamString						      */
/*									      */
/******************************************************************************/


ParamString::ParamString(const string &this_name, string this_value,  Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value(this_value) {
  set_flag=true;
}

ParamString::ParamString(const string &this_name, Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value() {
  set_flag=false;
}

string ParamString::get_value() const {
  
  if ((ParamString*)baseparamp!=this)
    return ((ParamString*)baseparamp)->get_value();
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value;
  }
}

void ParamString::my_set_with_string(const string &s, Boolean) {

  value=s;
  value.erase(remove(value.begin(), value.end(), '"'), value.end());
  
}

 
string ParamString::get_value_string() const {
  if ((ParamString*)baseparamp!=this)
    return ((ParamString*)baseparamp)->get_value_string();
  else {
    if (dirty_string_flag==false) return value_string; // Value string is up to date
    set_value_string(value);
    return value_string;
  }
}

Param* ParamString::return_parent_parameter(const string &this_name) {
  if (set_flag)
    return new ParamString(this_name, value, this);
  else
    return new ParamString(this_name, this);
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamEnumerated			       		      */
/*									      */
/******************************************************************************/


ParamEnumerated::ParamEnumerated(const string &this_name, vector<string> valid, const string &this_value,  Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value(this_value), valid_values(valid) {
  set_flag=true;
}

ParamEnumerated::ParamEnumerated(const string &this_name, vector<string> valid, Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value(), valid_values(valid) {
  set_flag=false;
}

ParamEnumerated::ParamEnumerated(const string &this_name, const char *valid[], const string &this_value,  Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value(this_value), valid_values() {
  make_valid_values(valid);
  set_flag=true;
}

ParamEnumerated::ParamEnumerated(const string &this_name, const char *valid[], Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value(), valid_values() {
  set_flag=false;
  make_valid_values(valid);
}


vector<string> ParamEnumerated::get_valid_values() const {
  if ((ParamEnumerated*)baseparamp!=this)
    return ((ParamEnumerated*)baseparamp)->get_valid_values();
  else {
    return valid_values;
  }									      
}

string ParamEnumerated::get_value() const {
  
  if ((ParamEnumerated*)baseparamp!=this)
    return ((ParamEnumerated*)baseparamp)->get_value();
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value;
  }
}

Integer ParamEnumerated::get_index() const {

  if ((ParamEnumerated*)baseparamp!=this)
    return ((ParamEnumerated*)baseparamp)->get_index();
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    
    vector<string>::const_iterator stringp = find(valid_values.begin(), valid_values.end(),value);

    return stringp-valid_values.begin();
  }
}

void ParamEnumerated::my_set_with_string(const string &s, Boolean) {

  value=s;
  value.erase(remove(value.begin(), value.end(), '"'), value.end());
  
  if (!is_valid(s))
    throw(CTKErrorGCE(__FILE__,__LINE__, "Invalid value for enumerated parameter: ", s.c_str()));

}

 
string ParamEnumerated::get_value_string() const {
  if ((ParamEnumerated*)baseparamp!=this)
    return ((ParamEnumerated*)baseparamp)->get_value_string();
  else {
    if (dirty_string_flag==false) return value_string; // Value string is up to date
    set_value_string(value);
    return value_string;
  }
}

Param* ParamEnumerated::return_parent_parameter(const string &this_name) {
  if (set_flag)
    return new ParamEnumerated(this_name, valid_values, value, this);
  else
    return new ParamEnumerated(this_name, valid_values, this);
}


Boolean  ParamEnumerated::is_valid(string a_name) {
  
  vector<string>::iterator stringp = find(valid_values.begin(), valid_values.end(), a_name);
  return (stringp!=valid_values.end());
}

void ParamEnumerated::make_valid_values(const char *values[]) {
  int i=0;
  while (strlen(values[i])>0) {
    valid_values.push_back(values[i]);
    ++i;
  }
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamIntVector		       			      */
/*									      */
/******************************************************************************/


ParamIntVector::ParamIntVector(const string &this_name, const vector<int> &this_value,  Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value(this_value) {
  set_flag=true;
}

ParamIntVector::ParamIntVector(const string &this_name,  Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value() {
  set_flag=false;
}

Integer ParamIntVector::get_value(int index) const {
  
  if ((ParamIntVector*)baseparamp!=this)
    return ((ParamIntVector*)baseparamp)->get_value(index);
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value[index];
  }
}

vector<int> ParamIntVector::get_vector() const {
  
  if ((ParamIntVector*)baseparamp!=this)
    return ((ParamIntVector*)baseparamp)->get_vector();
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value;
  }
}

Integer ParamIntVector::get_size() const {
  
  if ((ParamIntVector*)baseparamp!=this)
    return ((ParamIntVector*)baseparamp)->get_size();
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value.size();
  }
}

void ParamIntVector::my_set_with_string(const string &s, Boolean) {
  
  char buffer[CTK_CHAR_BUFFER_SIZE+1];
  char crap[CTK_CHAR_BUFFER_SIZE+1];
  Integer x;
  Integer bytes;
  
  string ss=s;
  clean_param_string(ss);
  const char *s_copy=ss.c_str();

  vector<int> new_value;
  
  while (sscanf(s_copy, "%s%n", buffer, &bytes) == 1) {
    int nread=sscanf(buffer,"%d%s", &x, crap);  

    if (nread==0 || (nread==2 && (strlen(crap)>0)))
      throw(CTKErrorGCE(__FILE__,__LINE__, "Invalid value for integer vector parameter: ", get_param_fullname().c_str(), s.c_str()));
    else  if (validate(x)==CTK_FAILURE)
      throw(BlockErrorPOR(__FILE__, __LINE__, get_param_fullname().c_str(), s.c_str()));
    else
      new_value.push_back(x);
    
    s_copy+=bytes;
  }
  value=new_value;
  
}


string ParamIntVector::get_value_string() const {
  if ((ParamIntVector*)baseparamp!=this)
    return ((ParamIntVector*)baseparamp)->get_value_string();
  else {
    if (dirty_string_flag==false) return value_string; // Value string is up to date

    // Rebuild value string before returning it
    value_string.resize(0);

    Integer length=0;
    UInteger i;
    for (i=0; i<value.size(); ++i) 
      length+=2+(int)log(fabs((double)value[i])+1);

    char *one_value= new Character[length+1];
    for (i=0; i<value.size(); ++i) {
      sprintf(one_value,"%d ",value[i]);
      value_string+=one_value;
    }
    delete[]one_value;
    
    dirty_string_flag=false;
    return value_string;
  }
}


Param* ParamIntVector::return_parent_parameter(const string &this_name) {
  if (set_flag)
    return new ParamIntVector(this_name, value, this);
  else
    return new ParamIntVector(this_name, this);
}



/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamFloatVector      				      */
/*									      */
/******************************************************************************/


ParamFloatVector::ParamFloatVector(const string &this_name, const vector<Float> &this_value,  Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value(this_value) {
  set_flag=true;
}

ParamFloatVector::ParamFloatVector(const string &this_name, Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value() {
  set_flag=false;
}

Float ParamFloatVector::get_value(int index) const {
  
  if ((ParamFloatVector*)baseparamp!=this)
    return ((ParamFloatVector*)baseparamp)->get_value(index);
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value[index];
  }
}

vector<Float> ParamFloatVector::get_vector() const {
  
  if ((ParamFloatVector*)baseparamp!=this)
    return ((ParamFloatVector*)baseparamp)->get_vector();
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value;
  }
}

Integer ParamFloatVector::get_size() const {
  
  if ((ParamFloatVector*)baseparamp!=this)
    return ((ParamFloatVector*)baseparamp)->get_size();
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value.size();
  }
}

void ParamFloatVector::my_set_with_string(const string &s, Boolean) {

  char buffer[CTK_CHAR_BUFFER_SIZE];
  char crap[CTK_CHAR_BUFFER_SIZE+1];
  Float x;
  Integer bytes;
  
  string ss=s;
  clean_param_string(ss);
  const char *s_copy=ss.c_str();

  vector<Float> new_value;
  
  while (sscanf(s_copy, "%s%n", buffer, &bytes) == 1) {
    int nread=sscanf(buffer,"%lf%s", &x, crap);
    if (nread==0 || (nread==2 && (strlen(crap)>0)))
      throw(CTKErrorGCE(__FILE__,__LINE__, "Invalid value for floating point vector parameter: ",  get_param_fullname().c_str(), s.c_str()));
     else  if (validate(x)==CTK_FAILURE)
       throw(BlockErrorPOR(__FILE__, __LINE__, get_param_fullname().c_str(), s.c_str()));
     else
      new_value.push_back(x);
    s_copy+=bytes;
  }
  value=new_value;

}


string ParamFloatVector::get_value_string() const {
  if ((ParamFloatVector*)baseparamp!=this)
    return ((ParamFloatVector*)baseparamp)->get_value_string();
  else {
    if (dirty_string_flag==false) return value_string; // Value string is up to date
    
    // Rebuild value string before returning it
    value_string.resize(0);
    char one_value[8];

    for (UInteger i=0; i<value.size(); ++i) {
      sprintf(one_value,"%5.3f ",value[i]);
      value_string+=one_value;
    }
    dirty_string_flag=false;
    return value_string;
  }
}


Param* ParamFloatVector::return_parent_parameter(const string &this_name) {
  if (set_flag)
    return new ParamFloatVector(this_name, value, this);
  else
    return new ParamFloatVector(this_name, this);
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamBoolVector	      				      */
/*									      */
/******************************************************************************/


ParamBoolVector::ParamBoolVector(const string &this_name, const vector<bool> &this_value,  Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value(this_value) {
  set_flag=true;
}

ParamBoolVector::ParamBoolVector(const string &this_name, Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value() {
  set_flag=false;
}

Boolean ParamBoolVector::get_value(int index) const {
  
  if ((ParamBoolVector*)baseparamp!=this)
    return ((ParamBoolVector*)baseparamp)->get_value(index);
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value[index];
  }
}

vector<bool> ParamBoolVector::get_vector() const {
  
  if ((ParamBoolVector*)baseparamp!=this)
    return ((ParamBoolVector*)baseparamp)->get_vector();
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value;
  }
}

Integer ParamBoolVector::get_size() const {
  
  if ((ParamBoolVector*)baseparamp!=this)
    return ((ParamBoolVector*)baseparamp)->get_size();
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value.size();
  }
}

void ParamBoolVector::my_set_with_string(const string &s, Boolean) {

  char buffer[CTK_CHAR_BUFFER_SIZE];
  
  string ss=s;
  clean_param_string(ss);
  const char *s_copy=ss.c_str();
  vector<bool> new_value;

  while (sscanf(s_copy, "%s", buffer) == 1) {
    if (*s_copy == '1' || *s_copy == 't' || *s_copy == 'T' || *s_copy == 'y' || *s_copy == 'Y') 
      new_value.push_back(true);
    else if (*s_copy == '0' || *s_copy == 'f' || *s_copy == 'F' || *s_copy == 'n' || *s_copy == 'N') 
      new_value.push_back(false);
    else {
      throw(CTKErrorGCE(__FILE__,__LINE__, "Invalid value for boolean vector parameter: ", s.c_str()));
    }
    s_copy+=strlen(buffer);
  }

  value=new_value;
  
}


string ParamBoolVector::get_value_string() const {
  if ((ParamBoolVector*)baseparamp!=this)
    return ((ParamBoolVector*)baseparamp)->get_value_string();
  else {
    if (dirty_string_flag==false) return value_string; // Value string is up to date

    // Rebuild value string before returning it
    value_string.resize(0);

    for (UInteger i=0; i<value.size(); ++i) 
      value_string+=(value[i]?"Yes ":"No ");
    
    dirty_string_flag=false;
    return value_string;
  }
}

Param* ParamBoolVector::return_parent_parameter(const string &this_name) {
  if (set_flag)
    return new ParamBoolVector(this_name, value, this);
  else
    return new ParamBoolVector(this_name, this);
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamStringVector	      				      */
/*									      */
/******************************************************************************/


ParamStringVector::ParamStringVector(const string &this_name, const vector<string> &this_value,  Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value(this_value) {
  set_flag=true;
}

ParamStringVector::ParamStringVector(const string &this_name, Param* this_subparamp/*=NULL*/): Param(this_name, this_subparamp), value() {
  set_flag=false;
}

string ParamStringVector::get_value(int index) const {
  
  if ((ParamStringVector*)baseparamp!=this)
    return ((ParamStringVector*)baseparamp)->get_value(index);
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value[index];
  }
}

vector<string> ParamStringVector::get_vector() const {
  
  if ((ParamStringVector*)baseparamp!=this)
    return ((ParamStringVector*)baseparamp)->get_vector();
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value;
  }
}

Integer ParamStringVector::get_size() const {
  
  if ((ParamStringVector*)baseparamp!=this)
    return ((ParamStringVector*)baseparamp)->get_size();
  else {
    if (set_flag==false) throw(ParamErrorPNS(__FILE__,__LINE__, get_param_pathname().c_str()));
    return value.size();
  }
}

void ParamStringVector::my_set_with_string(const string &s, Boolean) {

  char buffer[CTK_CHAR_BUFFER_SIZE];
  Integer bytes;
  
  string ss=s;
  clean_param_string(ss);
  const char *s_copy=ss.c_str();

  vector<string> new_value;

  while (sscanf(s_copy, "%s%n", buffer, &bytes) == 1) {
    new_value.push_back(string(buffer));
    s_copy+=bytes;
  }

  value=new_value;
  
}


string ParamStringVector::get_value_string() const {
  if ((ParamStringVector*)baseparamp!=this)
    return ((ParamStringVector*)baseparamp)->get_value_string();
  else {
    if (dirty_string_flag==false) return value_string; // Value string is up to date
    
    // Rebuild value string before returning it
    value_string.resize(0);

    for (UInteger i=0; i<value.size(); ++i) {
      value_string+=(value[i]);
      value_string+=", ";
    }
    dirty_string_flag=false;
    return value_string;
  }
}

Param* ParamStringVector::return_parent_parameter(const string &this_name) {
  if (set_flag)
    return new ParamStringVector(this_name, value, this);
  else
    return new ParamStringVector(this_name, this);
}



/* End of ctk_param.cpp */

