/******************************************************************************/
/*									      */
/*	ctk_pipe.hh	    						      */
/*									      */
/*	Class declarations for ctk_pipe.cpp				      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		        	      */
/*									      */
/******************************************************************************/

#ifndef CTK_PIPE_HH
#define CTK_PIPE_HH

#include <vector>

#include "ctk_local.hh"
#include "ctk_types.hh"  // For CTKVector and CTKSample 

/******************************************************************************/
/*									      */
/*	Parameters							      */
/*									      */
/******************************************************************************/

const Integer BUFFER_INITIAL_SIZE = 4096;  
const Integer MIN_EXPANSION = 4096;  



/******************************************************************************/
/*									      */
/*	CLASS NAME: CircularBufferr -Simple circular buffer for floats 	      */
/*									      */
/******************************************************************************/

template<class T> class CircularBuffer {
  
private:

  vector<T> data;
  UInteger capacity;     // Size of buffer
  UInteger free_space;   // Number of free bytes
  UInteger storage;      // Amount of data stored
  UInteger start;        // Data start pointer
  UInteger end;          // Data end pointer (i.e. first free position)

public:

  typedef typename vector<T>::iterator DataIt;
  typedef typename vector<T>::const_iterator DataConstIt;
  
  CircularBuffer(Integer size) {
    data=vector<T>(size);
    capacity=size;
    
    start=end=0;
    storage=0;
    free_space=capacity;
  }
  
  ~CircularBuffer() {
    //  cerr << "In CircularBuffer Destructor\n\r";
  }
  
  void clear_and_delete(Integer new_size=0) {
    vector<T> dummy;

    flush(dummy);

    for (unsigned int i=0; i<dummy.size(); ++i)
      delete dummy[i];
    
    if (new_size!=0) capacity=new_size;
    data.clear();
    data.resize(capacity);
    start=end=0;
    storage=0;
    free_space=capacity;
  }
  
  
  void clear(Integer new_size=0) {
    vector<T> dummy;

    flush(dummy);
    
    if (new_size!=0) capacity=new_size;
    data.clear();
    data.resize(capacity);
    start=end=0;
    storage=0;
    free_space=capacity;
  }
  
  UInteger get_storage() const{return storage;}
  
  Boolean put_back(UInteger n) {
    if (n>free_space) return false;
    storage+=n;
    free_space-=n;
    start=(start+capacity-n)%capacity;
    return true;
  }
  
  Boolean get(T &x) {
    if (storage==0) return false;
    
    x=data[start];
    
    start=(start+1)%capacity;
    
    --storage;
    ++free_space;
    return true;
  }
  
  Boolean put(const T &x) {
    if (free_space==0) this->expand(MIN_EXPANSION);
    
    data[end]=x;
    end=(end+1)%capacity;
    ++storage;
    --free_space;
    return true;
  }
  
  Boolean read(vector<T> &out_data) {
    UInteger n=out_data.size();
    
    if (n>storage) return false;
    
    DataIt odp = out_data.begin();
    for (UInteger i=start; i!=n+start; ++i) 
      *odp++=data[i%capacity];
    
    storage-=n;
    free_space+=n;
    start=(start+n)%capacity;
    return true;
  }
  
  Boolean write(const vector<T> &in_data, Integer pos=0, UInteger n=0) {

    if (n<=0)
      n=in_data.size();
    else {
      UInteger m;
      if ((m=(in_data.size()-pos))<n) n=m;
    }
    
    if (n>free_space) this->expand(n);

    DataConstIt idp=in_data.begin()+pos;
    for (UInteger i=end; i<end+n; ++i) 
      data[i%capacity]=*idp++;
        
    storage+=n;
    free_space-=n;
    end=(end+n)%capacity;
    return true;
  }
  
  Integer flush(vector<T> &out_data) {
    int flushed=storage;
    out_data.resize(storage);
    if (read(out_data)==false) {
      cerr << "CircularBuffer::flush - error during read\n\r";
      exit(-1);
    }
    
    return flushed;
  }
  
private:
  
 
  Boolean expand(Integer expansion) {
    
    expansion=(expansion<MIN_EXPANSION)?MIN_EXPANSION:expansion;

    Integer new_size=capacity+expansion;
    
    vector<T> new_buffer(new_size);

    //    for (UInteger i=0; i<storage; ++i) {
    //    new_buffer[i]=data[(start+i)%capacity];
    //  }

    DataIt new_bufferp=new_buffer.begin();
    for (UInteger i=start; i!=storage+start; ++i) {
      *new_bufferp++=data[i%capacity];
    }
    
    capacity=new_size;
    start=0; end=storage;
    free_space=capacity-storage;
    data=new_buffer;
    
    return true;
  }
  
};

/******************************************************************************/
/*									      */
/*	CLASS NAME: Pipe	         				      */
/*									      */
/******************************************************************************/

class Pipe {

private:

  Boolean initial_state;
  Boolean end_of_data;

public:
  
  Pipe();
  virtual ~Pipe(){};
  
  virtual void clear(Integer new_size=0)=0;
  virtual Boolean primed()=0;
  virtual UInteger get_storage() const =0;
  virtual Boolean put_back(Integer n) =0;

  Boolean get_end_of_data() const;
  Boolean is_primed_on_reset() const;
  void set_end_of_data();
  void make_primed_on_reset();

  virtual void reset();
  
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: FloatPipe	         				      */
/*									      */
/******************************************************************************/


class FloatPipe: public Pipe {
  
private:

  CircularBuffer<Float> *buffer;
  
public:

  FloatPipe(Integer nframes=BUFFER_INITIAL_SIZE);
  virtual ~FloatPipe();
  
  void clear(Integer new_size=0);

  void reset();

  Boolean primed();

  UInteger get_storage() const;
  Boolean put_back(Integer n);
  
  Boolean get(Float &x);
  Boolean put(const Float &x);
  
  Boolean read(vector<Float> &out_data);
  
  Boolean write(const vector<Float> &in_data, Integer pos=0, Integer n=0);
  
  Integer flush(vector<Float> &out_data);
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: FramePipe	         				      */
/*									      */
/******************************************************************************/


class FramePipe: public Pipe {
  
private:

  CircularBuffer<CTKVector*> *buffer;
  
public:

  FramePipe(Integer nframes=BUFFER_INITIAL_SIZE);
  virtual ~FramePipe();
  
  void clear(Integer new_size=0);

  void reset();

  Boolean primed();
  UInteger get_storage() const;
  Boolean put_back(Integer n);
  
  Boolean get(CTKVector* &x);
  Boolean put(CTKVector* x);
  Boolean read(vector<CTKVector*> &out_data);
  Boolean write(const vector<CTKVector*> &in_data, Integer pos=0, Integer n=0);
  Integer flush(vector<CTKVector*> &out_data);
};


#endif

/* End of ctk_pipe.hh */
