/******************************************************************************/
/*									      */
/*	ctk_reco.hh   			             		      */
/*									      */
/*	Class declarations for ctk_reco.cpp	              	      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			     	      */
/*									      */
/******************************************************************************/

#ifndef CTK_RECO_HH
#define CTK_RECO_HH

#include <sys/types.h>
#include <regex.h> /* Provides regular expression matching */

#include <cstdio>

#include <list>
#include <map>
#include <string>
#include <vector>

#include "ctk_local.hh"

#include "ctk_HMM_types.hh"

/******************************************************************************/
/*									      */
/*	CLASS NAME: RecoHypothesis		              	       	      */
/*									      */
/******************************************************************************/

string list_string_to_string(const list<string> &lstring);
list<string> string_to_list_string(const string &string);

class RecoHypothesis {

  friend Boolean compare_hypotheses(RecoHypothesis *hyp1, RecoHypothesis *hyp2 );
 
private:
  
  list<string> solution;    // The sequence of HMM labels
  HMMFloat score;      // The overall cost 
  list<int> bounds;   // The frame index of each HMM exit 
  list<vector<int> > states;  // The state sequence for each HMM visited
  string group_record;  // The fragments included (for multisource decoding)
  
public:
  
  RecoHypothesis(){}
  RecoHypothesis(const list<string> &a_solution, HMMFloat a_score, list<int> the_bounds, list<vector <int> > the_states, string groups);

  ~RecoHypothesis(){}
  
  const list<string> &get_solution_raw() const;
  const list<string> get_solution_source() const;
  Boolean matches_filter(regex_t *filter) const;
  string get_group_record() const;

  void write_all(FILE *file);  // Write all hypothesis details to file 
  void write_all_multisource(FILE *file, map<Integer, Integer> &number_to_label_map);  // Write all multisource decoding hypothesis details to file

  // Write hypothesis in TIMIT format e.g. lines with the format:
  //   start_sample end_sample HMM_label
  void write_TIMIT_format(FILE *file, int samples_per_frame) const;
  
private:
  void write_boundaries(FILE *file) const;
  void write_model_sequence(FILE *file) const;
  bool parse_model_name(const string &label, string &name1, string &name2) const;

  friend ostream& operator<< (ostream& out, const RecoHypothesis &hyp);
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: RecoStats   		              	       	      */
/*									      */
/******************************************************************************/

enum {NILDIR, DIAGDIR, VERTDIR, HORDIR};

struct GridPos {
  Integer score;
  Integer del;
  Integer ins;
  Integer subs;
  Integer hits;
  Integer dir;
};


class RecoStats {

private:
  
  list<string> labels;
  int nlabels;
  map<string, int> label_num;   // Mapping of label to row (or column) in confusion matrix
  
  int ntokens_ref;
  int ntokens_test;
  
  int hits;
  int dels;
  int ins;
  int subs;

  Float correctness;
  Float accuracy;

  vector<vector<int> > confusions;
  vector<int> deletions;
  vector<int> insertions;

public:
  RecoStats();
  RecoStats(const list<string> &labels);
  ~RecoStats();

  void calc_stats(const list<string> &test, const list<string> &ref);
  void print_stats(FILE *logfile);
  void print_confusions(FILE *logfile);

  void operator+=(const RecoStats &stats);
  
private:

  void commonConstructor();  // Code common to all constructors
  
  void align_strings(const list<string> &test_string, const list<string> &ref_string, vector<vector<GridPos> > *grid);
  void set_labels(const list<string> &some_labels);
  inline Float compute_correctness(int H, int N);
  inline Float compute_accuracy(int H, int I, int D, int S);
};


#endif

/* End of ctk_reco.hh */

