/******************************************************************************/
/*                                                                            */
/*      ctk_ro_file.hh                                                        */
/*                                                                            */
/*      Class declarations for ctk_ro_file.cpp                                */
/*                                                                            */
/*      Author: Jon Barker, Sheffield University                              */
/*                                                                            */
/*      CTK VERSION 1.3.5  Apr 22, 2007                              */
/*                                                                            */
/******************************************************************************/

#ifndef CTK_RO_FILE_HH
#define CTK_RO_FILE_HH

#include <cstdio>

#include <map>
#include <string>

#include "ctk_local.hh"

// The aim of the Read Once file system is to cut down the overhead of
// reading complex file types when processing long filelists with CTK.


// For example an HMM object is constructed from an HMM file during the
// reset() method of a decoder block. Its possible that a new HMM object
// will be required for each element in a file list, but normally the
// same HMMs will be used for every utterance and it would be wasteful
// to have to reread the same file. The read once file type prevents
// the rereading by keeping track of which files have been read and
// what objects were constructed from them. If a file has been read
// already then the object that it builds can be returned immediately.

// If the file has been modified since the last read, then it will either
// be read again (interactive mode) or not (batch mode). Batch mode means
// that experiments won't be ruined if the HMM file is editted (or deleted!)
// during the processing of a corpora. Interactive mode allows the user
// to run demos using the GUI where changes to the HMM file are reflected
// without having to restart the GUI session.

// Its all a bit hacky and horrible... 
// Objects that can be constructed from a read_once file have
// to inherit from ReadOnceFile. And the FileDescriptor class, which
// associates the objects with the file names from which they are built
// has to be modified for each type of ReadOnceFile added to the system...
// surely that shouldn't be necessary if the thing was designed properly :(

// June 13, 2002:
// The system has been modified so that if different blocks open the
// same read once file then the file will be read once per block.
// This had to be added when the GMM (which requires reading an HMM file)
// was separated from the decoder (which requires reading the same HMM file
// but which needs to maintain a separate HMM object).


/******************************************************************************/
/*									      */
/*	CLASS NAME: ctk_ro_file		                 	       	      */
/*									      */
/******************************************************************************/

// This class works in conjuction with:
// FileDescriptor  (see ctk_file_desc.hh)

class ReadOnceFile {

private:

  string filename_;

  
  FILE *ro_fp_;

public:
  
  static Boolean interactive_mode_;
  
  //
  //
  //
  
public:

  ReadOnceFile(){};
  ReadOnceFile(const string &filename);

  virtual ~ReadOnceFile();

  unsigned int count_lines();

protected:

  FILE *ro_fp(){return ro_fp_;}
  
};


static map<string, class FileDescriptor*> file_table;

void file_table_add_entry(class SetOfHMMs *rof, const string &a_filename, const string &blockname);
void file_table_add_entry(class HMM *rof, const string &a_filename, const string &blockname);

class ReadOnceFile *file_table_fetch_entry(const string &filename, const string &blockname, Boolean ignore_timestamp=false);
class ReadOnceFile *file_table_fetch_entry_ignoring_timestamp(const string &filename, const string &blockname);

bool file_table_delete_entry(const string &a_filename, const string &blockname);


#endif

/* End of ctk_ro_file.hh */
