/******************************************************************************/
/*									      */
/*	ctk_script.cpp	    		        	      		      */
/*									      */
/*	Implementation for CTKScriptElement:				      */
/*	      support for interpretting CTK scripts       	       	      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007	        		      */
/*						  			      */
/******************************************************************************/

#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_script.hh"


/******************************************************************************/
/*						 			      */
/*	CLASS NAME: CTKScriptElement 		             	       	      */
/*									      */
/******************************************************************************/

Boolean CTKScriptElement::is_a_comment() const{
  UInteger i=0;
  while (i<my_string.size() && isspace(my_string[i])) ++i;
  if (i==my_string.size()) return false;
  return  (my_string[i]=='#');
}

Boolean CTKScriptElement::is_a_help_comment() const{
  UInteger i=0;
  while (i<(my_string.size()-1) && isspace(my_string[i])) ++i;
  if (i==my_string.size()-1) return false;
  return  (my_string[i]=='#' && my_string[i+1]=='H');
}


Boolean CTKScriptElement::is_blank() const {
  UInteger i=0, length=my_string.size();
  while (i<length && isspace(my_string[i])) ++i;
  return (i==length);
}


// Remove and return the next word - here a word is a bit of non-whitespace surrounded by whitespace
string CTKScriptElement::get_word() {
  string word;

  UInteger cpos=0;

  // Skip over initial white space
  while (cpos<my_string.size() && isspace(my_string[cpos])) ++cpos;

  // Script element contains no more words - return empty string
  if (cpos==my_string.size()) return word; 

  while (cpos<my_string.size() && !isspace(my_string[cpos]))
    word+=my_string[cpos++];
  

  // Position cpos after white space that comes after the word
  while (cpos<my_string.size() && isspace(my_string[cpos])) ++cpos;  
  
  // Remove everything up to cpos
  my_string.erase(0, cpos);

  return word;
}


// Remove and return the next 'script word' -- Here the definition of 'word' has been tailored to meet the script syntax ...e.g. something enclosed in quotes or braces is considered a single word.
string CTKScriptElement::get_script_word() {
  string word;

  UInteger cpos=0;

  // Find 1st character of word
  while (cpos<my_string.size() && isspace(my_string[cpos])) ++cpos;

  // Script element contains no more words - return empty string
  if (cpos==my_string.size()) return word; 

  if (my_string[cpos]=='"') {
    // Returned quoted element
    ++cpos;
    while (cpos<my_string.size() && (my_string[cpos]!='"'))
      word+=my_string[cpos++];
    if (my_string[cpos]=='"')
      ++cpos;
    else {
      cerr << "Missing \" in script" << endl;
      throw(CTKError(__FILE__, __LINE__));
    }
  } else if (my_string[cpos]=='{') {
    // Return bracketed element
    ++cpos;
    while (cpos<my_string.size() && (my_string[cpos]!='}'))
      word+=my_string[cpos++];
    if (my_string[cpos]=='}')
      ++cpos;
    else {
      cerr << "Missing } in script" << endl;
      throw(CTKError(__FILE__, __LINE__));
    }
  } else {
    // Return word
    while (cpos<my_string.size() && isname(my_string[cpos]))
      word+=my_string[cpos++];
  }

  // Position cpos after space that follows word that was removed
  while (cpos<my_string.size() && isspace(my_string[cpos])) ++cpos;  

  // Remove word 
  my_string.erase(0, cpos);

  return word;
}


string CTKScriptElement::get_braces() {
  string word;
  strip_whitespace();
  if (my_string[0]=='(') {
    word+='(';
    UInteger cpos=1;
    while(cpos<my_string.size() && my_string[cpos]!=(')')) {
      if (my_string[cpos]=='(') {
	my_string.erase(0, cpos);
	cpos=0;
	word+=get_braces();
      } else if (my_string[cpos]=='"') {
	word+=my_string[cpos++];
	while (cpos<my_string.size() && my_string[cpos]!='"')
	  word+=my_string[cpos++];
      } else
	word+=my_string[cpos++];
    }
    if (cpos!=my_string.size()) {word+=')'; ++cpos;};
    my_string.erase(0, cpos);
  }
  return word;
}

Boolean CTKScriptElement::get_char(char x) {
  strip_whitespace();
  if (my_string[0]==x) {
    my_string.erase(0,1);
    return true;
  } else
    return false;
}

void CTKScriptElement::strip_whitespace() {
  UInteger cpos=0;
  while (isspace(my_string[cpos]) && cpos<my_string.size())
    ++cpos;
  my_string.erase(0,cpos);
}
  

Boolean CTKScriptElement::interpret_assignment(string &s1, string &s2)  {

  s1=get_script_word();
  
  if (get_char('=')==false) return true;   // If can't find '=' then return error

  s2=get_script_word();
  
  if (s1.size()==0) return true;

  return false;
}


void CTKScriptElement::replace_word(const string &in_word, const string &out_word) {
  Boolean start_ok, end_ok;
  
  UInteger start_pos=0;
  Integer length=in_word.size();
  while ((start_pos=my_string.find(in_word,start_pos))!=string::npos) {
    start_ok=((start_pos==0)||(isname(my_string[start_pos-1])==false));
    end_ok=((start_pos+length==my_string.size())||(isname(my_string[start_pos+length])==false));
    if (start_ok&&end_ok) {
      my_string.replace(start_pos, length, out_word);
    }
    start_pos+=1;
  }
} 


Boolean CTKScriptElement::interpret_block_parameters(vector<CTKScriptElement*> &assignment_list) {

  Character ch;
  
  UInteger length=my_string.size();

  if (length==0) return false;

  assignment_list.resize(0);
  Boolean in_array=false;
  CTKScriptElement *param;

  Boolean in_string=false;
  param=new CTKScriptElement(string("\0"));

  UInteger cpos=1;

  while (((ch=my_string[cpos])!=')') || (in_string==true) || (in_array==true)) {
    if (in_string && ch!='"')
      (*param)+=ch;
    else {
      if (ch=='"') {
	in_string=!in_string;
	(*param)+=ch;
      }
      else if (ch=='{') {
	in_array=true;
	(*param)+=ch;
      }
      else if (ch=='}') {
	in_array=false;
	  (*param)+=ch;
      }
      else if (ch==',' && !in_array) {
	assignment_list.push_back(param);
	in_string=false;
	  param=new CTKScriptElement("\0");
      }
      //      else if ((isspace(ch)==false)||(in_string==true))  // JPB 8/2/01 BUG FIX
      else if ((isspace(ch)==false)||(in_string==true)||(in_array==true))
	  (*param)+=ch;
    }
    if (++cpos==length) return true;
  }

  if (!in_array && !in_string) {
    assignment_list.push_back(param);
    return false;
  } else {
    return true;
  }

}





Boolean isname(Character c) {
  return isalnum(c)||(c=='_')||(c=='#')||(c==':')||(c=='.')||(c=='/')||(c=='$')||(c=='-');
}

void make_upper(string &s) {
  for (UInteger i=0; i<s.size(); ++i)
    s[i]=toupper(s[i]);
}


/* End of ctk_script.cpp */
