/******************************************************************************/
/*									      */
/*	ctk_shell.cpp	        			       		      */
/*									      */
/*	Implementation for ctk_shell            			      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		         	      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <unistd.h>
#include <sys/wait.h>  // For vfork, wait & execl

#ifdef _HAS_QT
#include <qapplication.h>
#include <qapp.h>
#include <qwidcoll.h>
#endif

#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_CTK_reader.hh"
#include "ctk_command_line.hh" 
#include "ctk_ro_file.hh"

#include "ctk_shell.hh"
#undef erase

#ifdef _HAS_CURSES
// Curses if ruquired for interactive shells
#define NOMACROS
#include <curses.h>
#undef erase
#undef move
#include "term.h"
#endif

/******************************************************************************/
/*									      */
/*	CLASS NAME: CTK_Shell			         	       	      */
/*									      */
/******************************************************************************/

CTK_Shell::CTK_Shell(istream *this_istr, Boolean interactive/*=false*/):istr(this_istr) {
  is_interactive=interactive;

  runtime_profiling=0;
  
#ifdef _HAS_CURSES
  // Need the curses library to run interactive shells
  
  if (is_interactive) {
    // Initialise history, command, and match buffers to be empty
    history.resize(0);
    buffer.resize(0);
    match_buffer.resize(0);
    
    historyp=history.end();
    bufferp=buffer.end();
    
    // Set up the terminal - clears the screen and displays the prompt
    
    initialise_terminal(">> ");
  }

#endif
  
}

CTK_Shell::~CTK_Shell() {
#ifdef _HAS_CURSES

  // Need the curses library to run interactive shells
  if (is_interactive)
    exit_terminal();

#endif
  
#ifdef _HAS_QT
  // wait for the last widget to be closed and then shut down Qt
  if (qApp!=NULL) {
    cerr << "Waiting for the last widget to close..." << endl;
    
    QObject::connect(qApp, SIGNAL(lastWindowClosed()), qApp, SLOT(quit()));
    
    qApp->exec();
    
    cerr << "The last widget closed." << endl;
    
    // shutdown the Qt engine
    
    qApp->quit();
    delete qApp;
   }

#endif
}

void CTK_Shell::set_runtime_profiling(Boolean xx) {
  runtime_profiling=xx;
}

void CTK_Shell::toggle_runtime_profiling() {
  set_runtime_profiling(!runtime_profiling);
}

void CTK_Shell::do_main(int argc, const char* argv[]) {

  CTKCommandLine command_line(string("-profile -param NULL -S NULL -block main"), argc, argv);
  
  string block_name=command_line.get_option_argument("block");

  runtime_profiling=command_line.option_is_present("profile");
  
  if (command_line.argument_count()==0)
    return;

  string file_name=command_line.get_argument(1);

  CTKReader script(file_name, command_line.get_arguments());
  
  Block *block=script.get_block(block_name);

  if (block!=NULL) {
    if (command_line.option_is_present("param")) {
      do {
	CTKScriptElement param = command_line.get_option_argument("param");
	string param_name, param_value;
	param.interpret_assignment(param_name, param_value);
	block->set_parameter(param_name, param_value.c_str());
      } while (command_line.next_occurrence("param"));
    }

    block->initialise();

    // Dry run through argument list to make sure that parameters are happy to be set to these values
    command_line.reset_to_first_argument_list();
    do {
      // Set parameters according to the current line of command arguments
      int n1, n2;
      if ((n1=(command_line.argument_count()-1))!=(n2=script.get_max_param_number())) {
	cerr << "Command script requires up to " << n2 << " command line parameters. "<< n1 << " were supplied." << endl;
	throw(CTKError(__FILE__, __LINE__));
      }

      setParametersFromCommandline(*block, command_line, script);

    } while (command_line.next_argument_list());
    command_line.reset_to_first_argument_list();


    // The main block processing loop
    Boolean mode=ReadOnceFile::interactive_mode_;
    do {
      // Set parameters according to the current line of command arguments
      setParametersFromCommandline(*block, command_line, script);

      block->process(runtime_profiling);

      ReadOnceFile::interactive_mode_=false;
    } while (command_line.next_argument_list());
    ReadOnceFile::interactive_mode_=mode;
    block->close_final_all();
    if (runtime_profiling)
      block->write_profiling_report(cout);
    delete block;
  } else {
    throw(ShellErrorCFB(__FILE__, __LINE__, block_name.c_str()));  
  }
  
}


Integer CTK_Shell::run() {
  
  if (!is_interactive) return 0;

#ifdef _HAS_CURSES

  // Need the curses library to run interactive shells

  const Integer MAX_ARGS=100;
  
  //  Character buffer[CTK_CHAR_BUFFER_SIZE];
  CTKScriptElement line;
  //  Pctk_shell_mem op;
  string op_string;
  int argc;
  const char *argv[MAX_ARGS];
  
  while (true) {
    //    cout << ">" << flush;
    //  istr->getline(buffer,CTK_CHAR_BUFFER_SIZE,'\n');
    // line=CTKScriptElement(string(buffer));

    string command=get_command();
    
    line=CTKScriptElement(command);
    
    if (!line.is_blank()) {
      if (command[0]=='!') {
	do_unix(command.c_str()+1);
      }
#ifdef _HAS_MATLAB
      else if (command[0]=='#') {
	do_matlab(command.c_str()+1);
      }
#endif
      else {
	//do CTK
	argv[0]="CTKScript";
	argc=1;
	while (line.size()>0) {
	  op_string=line.get_word();
	  if (op_string.size()==0) op_string="\"\"";  
	  argv[argc]=strdup(op_string.c_str());
	  ++argc;
	} 
	op_string=argv[1];

	
	if (op_string=="quit" || op_string=="exit") {

	  return 0;
	} else if (op_string=="version") {
	  cout << CTK_VERSION_STRING << endl;

	} else if (op_string=="profile") {
	  toggle_runtime_profiling();
	  if (runtime_profiling)
	    cout << "profiling: on" << endl;
	  else
	    cout << "profiling: off" << endl;
	    
	} else {
	  
	  try {
	    CTK_Shell::do_main(argc, argv);
	  }
#ifdef _HAS_MATLAB
	  // If whatever was typed doesn't match a ctk file name conatining a
	  // main block then treat it as a matlab command
	  catch (ShellErrorCFB  &error) {
	    do_matlab(command.c_str());
	  }
	  catch (FileErrorCFCF &error) {
	    do_matlab(command.c_str());
	  }
#endif
	  catch (CTKError &error) {
	    cerr << "CTK EXCEPTION CAUGHT BY DEFAULT HANDLER" << endl;
	    error.debug_print(); 
	  }
	}
      }
    }
  }

#else
  return 0;
#endif
  
}

#ifdef _HAS_MATLAB
void CTK_Shell::do_matlab(const char *command) {
  Block::dispatch_matlab_command(command);
}
#endif

void CTK_Shell::do_unix(const char *command) {
  if (vfork()) wait(0);
  else {
    // Make sure shell terminal is in NLCR mode - so newlines get a carriage return
    string qcommand=string("""stty onlcr; ")+command + '"';
    
    qcommand+='"';
    if (execl("/bin/sh", "/bin/sh", "-c", qcommand.c_str(), NULL)==-1) {
      perror(command);
      cerr << "\r";
      exit(-1);
    }
  } 
}

//   
//   Private methods
//


#ifdef _HAS_CURSES

// Methods for interactive shells: REQUIRES CURSES

void CTK_Shell::initialise_terminal(const char *a_prompt) {

  // Set up terminal
  window = initscr();

  //// Don't really understand this TERMCAP stuff
  ////  need to find out how to make this work.
  ////
  //  // Set up terminal to translate return as CR + LF
  //  char *x = tparm("nel", 1, 0, 0, 0, 0, 0, 0, 0, 0);
  //  if (x==NULL) cerr << "Error with tparm()\n";
  //  putp(x);

  clearok(window, true);
  cbreak();
  noecho();

  nonl();
  
  intrflush(stdscr,FALSE);
  keypad(stdscr,TRUE);
  scrollok(window, true);
  idlok(window, true);
  
  prompt=strdup(a_prompt);

  move(0,0);
  addstr(prompt);
  length=strlen(prompt);

}


void CTK_Shell::exit_terminal() {
  
  endwin();

}


//
// Get a command from the user
// This handles all the commandline editting and command history support
// It quits when a return key is hit and returns whatevers in the command
// buffer. The command history is updated
//

string CTK_Shell::get_command() {
#ifdef _HAS_CURSES
  
  string command;
  Boolean finish = false;
  
  while (1) {
    getyx(window, y, x);
    int ch= getch();
    getmaxyx(window, maxy, maxx);
    switch (ch) {
    case KEY_RIGHT:
      if (x<length) {
	move(y,x+1);
	++bufferp;
      }
      break;
    case KEY_LEFT:
      if (x>strlen(prompt)) {
	move(y,x-1);
	--bufferp;
      }
      break;
    case KEY_UP:
      if (historyp!=history.begin()) {
	vector<string>::iterator hp;
	CTKStatus status=search_back(hp);
	if (status==CTK_SUCCESS) {
	  historyp=hp;
	  replace_line(historyp->c_str(), prompt);
	}
      }
      break;
    case KEY_DOWN:
      if (historyp!=history.end() && historyp!=history.end()-1) {
	vector<string>::iterator hp;
	CTKStatus status=search_forward(hp);
	if (status==CTK_SUCCESS) {
	  historyp=hp;
	  replace_line(historyp->c_str(), prompt);
	}
      }
      break;
    case 8:   // KEY_BACKSPACE: ??
    case 127:  // KEY_DC: ??
      if (x>strlen(prompt)) {
	move(y,x-1);
	delch();
	length--;
	bufferp=buffer.erase(bufferp-1);
	match_buffer=buffer;
      }
      break;
    case '\r':
      break;
    case '\n':
      move(y,length);
      if (y==maxy) {
	addch('\r');
       	scrl(1);
      } else {
	addch('\n');
      }
      
      // This line is necessary if terminal is in LF->CRLF mode
      if (*(buffer.end()-1)=='\r') {buffer.erase(buffer.size()-1);}
      
      command=buffer;
      
      refresh();
      
      // Add command to command history if it is not an empty command
      // and it is different to the last command or the history is empty
      
      if (buffer.size()!=0 && (history.size()==0 || buffer!=*(history.end()-1))) {
	history.push_back(buffer);
      }
      historyp=history.end();
      
      finish=true;
      buffer.resize(0);
      match_buffer.resize(0);
      bufferp=buffer.end();
      length=strlen(prompt);
      addstr(prompt);
      break;
    default:
      bufferp=buffer.insert(bufferp,ch)+1;
      match_buffer=buffer;
      insch(ch);
      move(y,x+1);
      ++length;
    }
    if (finish) break;   
  }

#endif
  
  return command;
  
}


// Replace the contents of the command line buffer
void CTK_Shell::replace_line(const char *new_line, const char *prompt) {
  move(y,strlen(prompt));
  for (UInteger i=0; i<buffer.size(); ++i)
    addch(' ');
  move(y,strlen(prompt));

  buffer=string(new_line);
  bufferp=buffer.end();
  char *buffer_str=strdup(buffer.c_str());
  addstr(buffer_str);
  length=strlen(prompt)+strlen(buffer.c_str());
  move(y, length);
  delete buffer_str;
}

CTKStatus CTK_Shell::search_back(vector<string>::iterator &back) {
  Integer bsize=match_buffer.size();
  if (bsize==0) {
    back=historyp-1;
    return CTK_SUCCESS;
  }
  vector<string>::iterator historyp_copy=historyp-1;
  while (historyp_copy>=history.begin()) {
    if (strncmp(match_buffer.c_str(), historyp_copy->c_str(), bsize)==0) {
      back=historyp_copy;
      return CTK_SUCCESS;
    }
    historyp_copy--;
  }

  return CTK_FAILURE;
}


CTKStatus CTK_Shell::search_forward(vector<string>::iterator &back) {
  Integer bsize=match_buffer.size();
  if (bsize==0) {
    back=historyp+1;
    return CTK_SUCCESS;
  }
  
  vector<string>::iterator historyp_copy=historyp+1;
  while (historyp_copy<history.end()) {
    if (strncmp(match_buffer.c_str(), historyp_copy->c_str(), bsize)==0) {
      back=historyp_copy;
      return CTK_SUCCESS;
    }
    ++historyp_copy;
  }

  return CTK_FAILURE;
}

// #endif for #ifdef _HAS_CURSES
#endif


/* End of ctk_shell.cpp */
