/******************************************************************************/
/*									      */
/*	ctk_socket.cpp	    						      */
/*									      */
/*	Implemenation of InputSocket and OutputSocket classes		      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007	         		      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <cstdio>

#if defined _DEBUG
#include <cmath>    // For isnan() in check_sample(Float)
#endif

#include <functional>
#include <algorithm>

#include "boost/iterator_adaptors.hpp"

#include "ctk_local.hh"

#include "ctk_function_classes.hh"
#include "ctk_block.hh"
#include "ctk_pipe.hh"
#include "ctk_data_descriptor.hh"

#include "ctk_socket.hh"



const int MAX_NUMBER_OF_SOCKETS = 8;

/******************************************************************************/
/*									      */
/*	CLASS NAME: SocketList	         				      */
/*									      */
/******************************************************************************/

  
SocketList::SocketList(Block *block):sockets(0) {
  configurable=false;
  parent_block=block;
  dirty_flag=false;
  min_num=1;
  max_num=MAX_NUMBER_OF_SOCKETS;
}

SocketList::~SocketList() {
  sequence_delete(sockets.begin(), sockets.end());
}

Socket* SocketList::operator[](int n) const {return sockets[n];}

Integer SocketList::size() const {return sockets.size();}
Integer SocketList::num_optional() const {return count_if(sockets.begin(), sockets.end(), mem_fun(&Socket::optional));}
Integer SocketList::num_non_optional() const {return sockets.size()-num_optional();}

Integer SocketList::get_min_num() const {return min_num;}
Integer SocketList::get_max_num() const {return max_num;}


Boolean SocketList::is_configurable() const {return configurable;}
Boolean SocketList::get_dirty_flag() const {return dirty_flag;}

Block* SocketList::get_parent_block(){return parent_block;}

Boolean SocketList::set_configurable(Boolean this_configurable, Integer a_min_num, Integer a_max_num) {
  if (size()!=0) return true; // can't change status once sockets have been added

  min_num=a_min_num;
  max_num=a_max_num;
  
  configurable=this_configurable;
  
  return false;
}

Integer SocketList::count_contiguous_connected_sockets() {
  Integer count=0;

  for (SocketIt socketpp=sockets.begin(); socketpp!=sockets.end(); ++socketpp) {
    if ((*socketpp)->connected())
      ++count;
    else break;
  }

  return count;
}


Boolean SocketList::set_description(const string &name, const string &desc) {
  SocketIt socketp=find_if(sockets.begin(), sockets.end(), bind2nd(mem_fun(&Socket::is_name),name));
  if (socketp!=sockets.end()) {
    (*socketp)->set_description(desc);
    return true;
  } else
    return false;
}

Boolean SocketList::set_optional(const string &name) {
  SocketIt socketp=find_if(sockets.begin(), sockets.end(), bind2nd(mem_fun(&Socket::is_name),name));
  if (socketp!=sockets.end()) {
    if (socketp==sockets.begin()) {
      cerr << "Warning: Cannot make first input socket an optional input (" << (*socketp)->get_socket_fullname() << ")" << endl;
      return false;
    }
    (*socketp)->set_optional();
    return true;
  } else
    return false;
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: InputSocketList	      				      */
/*									      */
/******************************************************************************/

  
InputSocketList::InputSocketList(Block *block):SocketList(block){};


void InputSocketList::configure(UInteger n) {

  char new_name[CTK_CHAR_BUFFER_SIZE];

  if (is_configurable()==false) {
    cerr << "Internal error - ctk_socket: attempting to configure non-configurable input" << endl;
    exit(-1);
  }

  if (n<sockets.size()) {
    sequence_delete(sockets.begin()+n, sockets.end());
    sockets.resize(n);
  } else {
    for (UInteger i=sockets.size(); i<n; ++i) {
      sprintf(new_name,"in%d",i+1);
      sockets.push_back(new InputSocket(new_name, get_parent_block()));
    }
  }
  set_dirty_flag();

}



//Socket* InputSocketList::operator[](int n) const {return sockets[n];}

InputSocket *InputSocketList::add_socket(const string &this_name, Block *this_block, Socket* this_subsocketp) {
  InputSocket *socket=new InputSocket(this_name, this_block, this_subsocketp);
  sockets.push_back(socket);
  return socket;
}


void InputSocketList::add(Socket &this_socket, Block *this_block, Socket* this_subsocketp) {
  sockets.push_back(this_socket.clone(this_block, this_subsocketp));
}


ostream& operator<< (ostream& out, const InputSocketList& sockets) {

  for (vector<Socket*>::const_iterator socketp=sockets.begin(); socketp!=sockets.end(); ++socketp) 
    out << **socketp;    
  return out;
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: OutputSocketList	      				      */
/*									      */
/******************************************************************************/


OutputSocketList::OutputSocketList(Block *block):SocketList(block){};


void OutputSocketList::configure(UInteger n) {

  char new_name[CTK_CHAR_BUFFER_SIZE];

  if (is_configurable()==false) {
    cerr << "ctk_socket: attempting to configure non-configurable output" << endl;
    exit(-1);
  }

  if (n<sockets.size()) {
    sequence_delete(sockets.begin()+n, sockets.end());
    sockets.resize(n);
  } else {
    for (UInteger i=sockets.size(); i<n; ++i) {
      sprintf(new_name,"out%d",i+1);
      sockets.push_back(new OutputSocket(new_name, get_parent_block()));
    }
  }
  set_dirty_flag();

}



OutputSocket *OutputSocketList::add_socket(const string &this_name, Block *this_block, Socket* this_subsocketp) {
  OutputSocket *socket = new OutputSocket(this_name, this_block, this_subsocketp);
  sockets.push_back(socket);
  return socket;
}


void OutputSocketList::add(Socket &this_socket, Block *this_block, Socket* this_subsocketp) {
   sockets.push_back(this_socket.clone(this_block, this_subsocketp));
}

ostream& operator<< (ostream& out, const OutputSocketList& sockets) {

  for (vector<Socket*>::const_iterator socketp=sockets.begin(); socketp!=sockets.end(); ++socketp) 
    out << **socketp;    
  return out;
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: Socket	         				      */
/*									      */
/******************************************************************************/

Socket::Socket(const string  &this_name, Block *this_block, Socket *this_subsocketp):name(this_name), description(this_name), blockp(this_block), baseblockp(NULL), data_desc(NULL), supersocketp(NULL), subsocketp(this_subsocketp), basesocketp(NULL),  remote_socket_name(string()), is_a_subsocket(false), is_optional(false), pipepF(NULL), pipepS(NULL), remote_socketp(NULL)   {

  baseblockp=((subsocketp==NULL)?blockp:subsocketp->baseblockp);
  basesocketp=((subsocketp==NULL)?this:subsocketp->basesocketp);

  if (subsocketp!=NULL) {
    subsocketp->is_a_subsocket=true;
    subsocketp->supersocketp=this;
    // Inherit the optional status of the subsocket
    if (subsocketp->optional())
      is_optional=true;
  }
}

void Socket::socket_copy(Socket *socket) { 
  is_optional=socket->optional();
  name=socket->get_name();
  description=socket->get_description();
  remote_socket_name=socket->get_remote_socket_name();

  data_desc=NULL;
  if (socket->data_desc!=NULL) {
    data_desc=new DataDescriptor(*socket->data_desc);
  }

}


Socket::~Socket() {
  delete data_desc;
}

void Socket::delete_pipe() {
  if (subsocketp==NULL) {  // base socket is responsible for deleting pipe
    if (pipepF!=NULL) delete(pipepF); 
    if (pipepS!=NULL) delete(pipepS); 
    pipepF=NULL;
    pipepS=NULL;
  }
}

Boolean Socket::set_description(const string &desc) {

  description=desc;
  return true;
}

Boolean Socket::is_name(string aname) const {
  return (aname==name);
}

Boolean Socket::am_i_called(string aname) const {
  return (aname==get_socket_name());
}

Boolean Socket::is_my_fullname(string aname) const {
  return (aname==get_socket_fullname());
}

Boolean Socket::am_i_a_subsocket() const {return is_a_subsocket;}

Block *Socket::get_baseblock() const {
  return baseblockp;
}

Block *Socket::get_block() const {
  return blockp;
}

Socket *Socket::get_basesocketp() const {
  return basesocketp;
} 

Pipe *Socket::get_pipep() const {
  if (data_desc->is_sample_data())
    return pipepS;
  else
    return pipepF;
}

FloatPipe *Socket::get_pipepS() const {return pipepS;}
FramePipe *Socket::get_pipepF() const {return pipepF;}

string Socket::get_socket_name() const {

  string fullname=blockp->prefix_object_name(name);

  return fullname;
};

string Socket::get_socket_fullname() const {

  string fullname=blockp->prefix_object_fullname(name);

  return fullname;
};

string Socket::get_subsocket_name() const {
  if (subsocketp==NULL) string();
  return subsocketp->get_socket_name();
}

string Socket::get_remote_socket_name() const {
  if (remote_socketp==NULL) return string();
  return remote_socketp->get_socket_name();
}

Block *Socket::get_remote_baseblock() const {return remote_socketp->get_baseblock();} 

Block *Socket::get_remote_block() const {return remote_socketp->get_block();} 

const Socket *Socket::get_remote_socket() const {return remote_socketp;}
const Socket *Socket::get_super_socket() const {return supersocketp;}

Boolean Socket::connected() const {return (pipepS!=NULL) && (pipepF!=NULL);}


Boolean Socket::connect(Socket *this_socketp, FramePipe *this_pipepF, FloatPipe *this_pipepS) {

  if (remote_socketp!=NULL) return false;
  
  remote_socketp=this_socketp;
  pipepF=this_pipepF;
  pipepS=this_pipepS;
  
  if (subsocketp!=NULL) return (subsocketp->connect(this_socketp, this_pipepF, this_pipepS));
  
  return true;
}

Boolean Socket::disconnect(Socket *this_socketp) {

  //  if (remote_socketp!=this_socketp) return false;
  
  remote_socketp=NULL;
  //  delete pipepF;
  //  delete pipepS;
  
  pipepF=NULL;
  pipepS=NULL;

  
  if (subsocketp!=NULL) return (subsocketp->disconnect(this_socketp));  // JON - Problem here
  
  return true;
}

void Socket::set_data_descriptor(const DataDescriptor *this_data_desc) {
  if (get_basesocketp()!=this) {
    get_basesocketp()->set_data_descriptor(this_data_desc);
  } else {
    if (data_desc!=NULL) {
      delete data_desc;
    }
    data_desc=this_data_desc;
  }
}

const DataDescriptor *Socket::get_data_descriptor() {
  if (get_basesocketp()!=this) {
    return get_basesocketp()->get_data_descriptor();
  } else return data_desc;
}


ostream& operator<< (ostream& out, const Socket &socket) {
  out << " " << socket.get_socket_name();
  if (!socket.get_remote_socket_name().empty())
    out << "(<--" << socket.get_remote_socket_name() << ") ";
  
  return out;
}

#if defined _DEBUG
void Socket::check_sample(Float sample) const {
  if (isnan(sample)) {
    cerr << "WARNING! Sample is NaN at socket " << get_socket_fullname() << "\n";
  }
}
#endif

/******************************************************************************/
/*									      */
/*	CLASS NAME: OutputSocket	      				      */
/*									      */
/******************************************************************************/

OutputSocket::OutputSocket(const string &this_name, Block *this_block, Socket* this_subsocketp/*=NULL*/):Socket(this_name, this_block, this_subsocketp){};
  
Socket* OutputSocket::clone(Block *this_block, Socket* this_subsocketp) {
  OutputSocket *sp = new OutputSocket(get_name(), this_block, this_subsocketp);
  sp->copy(this);
  return sp;
}


OutputSocket::~OutputSocket() {
  delete_pipe();
}

void OutputSocket::copy(OutputSocket *socket) {
  Socket::socket_copy(socket);
}

void OutputSocket::set_end_of_data() const {
  if (connected()) {
    get_pipepF()->set_end_of_data();
    get_pipepS()->set_end_of_data();
  }
}

void OutputSocket::reset() {
  if (connected()) {
    get_pipepF()->reset();
    get_pipepS()->reset();
  }
}

void OutputSocket::fire(){if (remote_socketp!=NULL) remote_socketp->fire();}


Boolean OutputSocket::make_connection(Socket *this_input_socket) {

  Boolean status;
  
  FloatPipe *this_pipepS=new FloatPipe(this_input_socket->get_nframes());
  FramePipe *this_pipepF=new FramePipe(this_input_socket->get_nframes());

  status=connect(this_input_socket,this_pipepF,this_pipepS)&&this_input_socket->connect(this, this_pipepF,this_pipepS);

  return status;
}


Boolean OutputSocket::break_connection(Socket *this_input_socket) {

  Boolean status;

  status=disconnect(this_input_socket)&&this_input_socket->disconnect(this);

  return status;
}






Boolean OutputSocket::put_sample(Float x) const {

#if defined _DEBUG
  check_sample(x);
#endif
  
  FloatPipe* pp=get_pipepS();
  if (pp!=NULL)
    return pp->put(x);
  else
    return false;
}


Boolean OutputSocket::put_vector(CTKVector *x) const {

#if defined _DEBUG
  for (vector<Float>::iterator xp=x->begin(), x_end=x->end(); xp!=x_end; ++xp)
    check_sample(*xp);
#endif

  if (x->size()!=1) {
    FramePipe* pp=get_pipepF();
    if (pp!=NULL)
      return pp->put(x);
    else {
      delete x;
      return false;
    }
  } else {
    Float xx=(*x)[0];
    delete x;
    return put_sample(xx);
  }
}


Boolean OutputSocket::write_sample(const CTKVector &x, Integer pos, Integer n) const {
  FloatPipe* pp=get_pipepS();
  if (pp!=NULL)
    return pp->write(x, pos, n);
  else
    return false;
}


Boolean OutputSocket::write_vector(const vector<CTKVector*> &x, Integer pos/*=0*/, Integer n/*=0*/) const {
  FramePipe* pp=get_pipepF();
  if (pp==NULL) {
    // Pipe not connected so destroy data
    for (UInteger i=0; i<x.size(); ++i)
      if (x[i]!=NULL) delete x[i];
    return false;
  } else {
    if (x[0]->size()!=1)
      return pp->write(x, pos, n);
    else {
      // Vectors have size 1 - they should be written to the sample pipe
      CTKVector *samples = new CTKVector(x.size());
      
      CTKVector::iterator svp=samples->begin();
      for (vector<CTKVector*>::const_iterator xp=x.begin(); xp!=x.end(); ++xp, ++svp) {
	*svp= (**xp)[0];
	delete *xp;
      }
      bool status=get_pipepS()->write(*samples, pos, n);
      delete samples;
      return status;
      
    }
  }
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: InputSocket	         				      */
/*									      */
/******************************************************************************/


InputSocket::~InputSocket() {

}

InputSocket::InputSocket(const string &this_name, Block *this_block, Socket* this_subsocketp/*=NULL*/):Socket(this_name, this_block, this_subsocketp){
  nframes=1;
  fired=false;
  initial_state=false;
};

Socket* InputSocket::clone(Block *this_block, Socket* this_subsocketp) {
  InputSocket *sp = new InputSocket(get_name(), this_block, this_subsocketp);
  sp->InputSocket::copy(this);
  return sp;
}

void InputSocket::copy(InputSocket *socket) {
  Socket::socket_copy(socket);
  nframes=socket->nframes;
  fired=socket->fired;
  initial_state=socket->initial_state;
}

void InputSocket::fire() {
  if (get_basesocketp()!=this)
    get_basesocketp()->fire();
  else fired=true;
}

void InputSocket::reset_fire() {

  if (get_basesocketp()!=this)
    get_basesocketp()->reset_fire();
  else fired=false;
}

Boolean InputSocket::has_fired() const {
  if ((InputSocket*)get_basesocketp()!=this)
    return get_basesocketp()->has_fired();
  else return fired;
}

void InputSocket::make_primed_on_reset(){
  if (get_basesocketp()!=this)
    get_basesocketp()->make_primed_on_reset();
  else {
    initial_state=true;
    get_pipepS()->make_primed_on_reset();
    get_pipepF()->make_primed_on_reset();
  }
}

UInteger InputSocket::get_nframes() const {
  if ((InputSocket*)get_basesocketp()!=this)
    return get_basesocketp()->get_nframes();
  else return nframes;
}

void InputSocket::reset(){
  if (get_basesocketp()==NULL)
    {cerr<<"Internal Error - NULL basesocket!" << endl; exit(-1);}

  if (get_basesocketp()!=this)
    get_basesocketp()->reset();
  else {fired=initial_state;}
}

// If the pipep==NULL then the socket is not connected
//     - therefore it must be an optional one and data_ready should return true
Boolean InputSocket::data_ready() const {
  return (!connected() || get_pipep()->get_storage()>=get_nframes());
}

Boolean InputSocket::get_end_of_data() const {return get_pipep()->get_end_of_data();}


// Pull data descriptor from previous output socket and use it as the descriptor for this socket ... if the previous output socket has no data descriptor then use the default instead.

const DataDescriptor * InputSocket::fetch_descriptor(const DataDescriptor *default_dd /* = NULL */) {
  if (connected()) {
    if (remote_socketp!=NULL) {
      const DataDescriptor *dd=remote_socketp->get_data_descriptor();
      if (dd!=NULL) {
	set_data_descriptor(new DataDescriptor (*dd));
	return dd;
      } else {
	// If the remote dd was not set, then use the default_dd if provided, else signal an error
	if (default_dd==NULL) {
	  cerr << "Internal error - InputSocket:fetch_descriptor has fetched a NULL data desc pointer - and no default was provided." << endl;
	  cerr << "Socket name= " << get_socket_fullname() << endl;
	  exit(-1);
	} else {
	  set_data_descriptor(new DataDescriptor(*default_dd));
	  return default_dd;
	}
      }
      
    } else {
      cerr << "Internal error - InputSocket:fetch_descriptor has a NULL remote socketp" << endl;
      exit(-1);
    }
  }

  return NULL;
  
}


Boolean InputSocket::put_back(Integer n) const {return get_pipep()->put_back(n);}


// Return true on success else false if the buffer is empty
Boolean InputSocket::get_sample(Float &data) const {return (get_pipepS())->get(data);}


// Get the data as a vector -
// Works regardless of whether data has been passed as a vector pointer or a float
// but InputSocket::get_sample will be more efficient if the data is known to be sample data

Boolean InputSocket::get_vector(CTKVector* &data) const {
  if (get_pipepF()->get(data)) return true;
  else {
    data=new CTKVector(1);
    return get_pipepS()->get((*data)[0]);
  }
}

Boolean InputSocket::read_sample(CTKVector &out_data) const {return (get_pipepS())->read(out_data);}

Boolean InputSocket::read_vector(vector<CTKVector*> &in_data) const {
  if (get_pipepF()->read(in_data)) return true;
  else {
    // Trying to read vectors from the sample pipe...
    // Must cast each sample into a vector. 
    CTKVector *samples= new CTKVector(in_data.size());
    if (!(get_pipepS()->read(*samples))) return false;
    CTKVector::iterator svp=samples->begin();
    for (vector<CTKVector*>::iterator idp=in_data.begin(); idp!=in_data.end(); ++idp, ++svp) 
      *idp=new CTKVector(1, *svp);
    delete samples;
    return true;
  }
  
}


Integer InputSocket::flush_sample(CTKVector &out_data) const {return (get_pipepS())->flush(out_data);}

Integer InputSocket::flush_vector(vector<CTKVector*> &in_data) const {

  Integer x;
  
  if ((x=get_pipepF()->flush(in_data))!=0) return x;
  else {
    // Trying to flush vectors from the sample pipe...
    // Must cast each sample into a vector. 
    CTKVector *samples= new CTKVector;
    x=get_pipepS()->flush(*samples);
    in_data.resize(x);
    CTKVector::iterator svp=samples->begin();
    for (vector<CTKVector*>::iterator idp=in_data.begin(); idp!=in_data.end(); ++idp, ++svp) 
      *idp=new CTKVector(1, *svp);
    delete samples;
    return x;
  }

}


/* End of ctk_socket.cpp */
 
