//Copyright Simon Tokumine 2003. Freely distributable under the GNU General Public Licence

import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.ParseException;
import java.util.ArrayList;
import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.JFormattedTextField;
import javax.swing.JPanel;
import javax.swing.KeyStroke;
import javax.swing.text.NumberFormatter;

/**
 * @author 	S. Tokumine
 * @date	19-Aug-2003
 * @time	16:13:47
 * @file	NumberInput.java
 * 
 * @desc this class gleans info from the user about the
 * number of colours wanted in the gradient
 *
 */
public class NumberInput extends JPanel implements PropertyChangeListener {

	private ColorPicker colorPicker; //parent
	private JFormattedTextField textField; //area to enter number
	private java.text.NumberFormat numberFormat; //format of number (2-10)
	private NumberFormatter formatter; //as above
	private ArrayList colorList; //list of colours
	private StatusBar statusBar;
	private PresetColor presetColor;

	/**
	 * constructor sets up links, creates textareas, and borders
	 */
	public NumberInput(
		ColorPicker colorPicker_in,
		ArrayList colorList_in,
		StatusBar statusBar_in) {
		super();

		//set up link to color picker for initialise
		colorPicker = colorPicker_in;
		colorList = colorList_in;

		//link up to status bar
		statusBar = statusBar_in;

		//Create the formatted text field and its formatter.
		numberFormat = java.text.NumberFormat.getIntegerInstance();
		formatter = new NumberFormatter(numberFormat);

		textField = new JFormattedTextField(formatter);
		textField.setColumns(8); //get some space
		textField.setPreferredSize(new Dimension(150, 20));
		initText();

		//add text box to the panel
		setLayout(new FlowLayout());
		add(textField);

		//Add border around the select panel.
		setBorder(
			BorderFactory.createCompoundBorder(
				BorderFactory.createTitledBorder(
					"1) Set number of colours in gradient (2-10): "),
				BorderFactory.createEmptyBorder(0, 0, 0, 0)));
	}

	public void setPreset(PresetColor presetColor_in) {
		presetColor = presetColor_in;
	}

	//initialise text areas listeners
	private void initText() {
		//text field
		formatter.setMinimum(new Integer(2));
		formatter.setMaximum(new Integer(10));
		textField.setValue(new Integer(0));
		//React when the user presses Enter.
		textField.getInputMap().put(
			KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0),
			"check");
		textField.getActionMap().put("check", new AbstractAction() {
			public void actionPerformed(ActionEvent e) {
				if (!textField.isEditValid()) { //The text is invalid.
					Toolkit.getDefaultToolkit().beep();
					textField.selectAll();
				} else
					try { //The text is valid,
						textField.commitEdit(); //so use it.
					} catch (java.text.ParseException exc) {
					}
			}
		});

		textField.addPropertyChangeListener(this);
	}

	/**
	 * accessor method for the preset colors
	 * this allows the preset colours to be edited
	 */
	public void setNumber(int number) {
		if (number >= 2 && number <= 10) {
			textField.setText(Integer.toString(number));
			try {
				textField.commitEdit();
			} catch (ParseException e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * when a number is entered, wipe the colourlist, and reset it to be full of 
	 * white colours up to the number entered. Set up the combobox, and disable the presets.
	 */
	public void propertyChange(PropertyChangeEvent e) {
		if ("value".equals(e.getPropertyName())) {
			Number value = (Number) e.getNewValue();
			if (colorPicker != null && value != null) {
				colorList.clear();
				for (int i = 0; i < value.intValue(); i++) {
					colorList.add(i, Color.WHITE);
				}
				statusBar.setStatus(
					"Number of colors in Gradient="
						+ value
						+ ". Set to white.");
				colorPicker.initPicker(value.intValue());
				presetColor.resetButtons();
			}
		}
	}
}
