//Copyright Simon Tokumine 2003. Freely distributable under the GNU General Public Licence

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.awt.image.MemoryImageSource;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.IOException;
import java.net.URL;
import java.util.Hashtable;
import java.util.Timer;
import java.util.TimerTask;
import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.KeyStroke;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.text.NumberFormatter;

/**
 * @author 	S. Tokumine
 * @date	24-Jun-2003
 * @time	19:37:58
 * @file	SlidePanel.java
 * 
 * @desc 	This class encapsulates the control section of the E3D
 * 			The JSliders value controls the current timestep displayed, and all
 * 			redraw operations occur through it. It is the start of the render pipeline.
 * 			The VCR controls all act by manipulating the slider automatically
 * 
 * 			The slidepanel also is in control of 3D vis initialisation
 * 			
 *
 */

public class SlidePanel
	extends JPanel
	implements MouseListener, PropertyChangeListener, ChangeListener, WindowListener {

	private SliceList sliceList; //list of timesteps
	private TextPanel textPanel; //output panel
	private JSlider slider; //slider control
	private JFormattedTextField textField; //not needed any more
	private java.text.NumberFormat numberFormat; //not needed anymore
	private NumberFormatter formatter; //not needed anymore
	private JButton go3D, begng, end, rew, ffw, play, stop; //vcr buttons
	private MainPanel mainPanel; //iamges
	private StatusBar statusBar;
	private Frame frame = null;
	private Image height2D, color2D; //two images to be displayed
	private boolean imageAvailiable = false;
	private ColorChooser colorChooser;
	private ImagePanel panel3D;
	private TeVi tevi = null;
	public Timer playTimer; //timer threads for the buttons
	private boolean changed = true; //not used now - used for synchronising 3d vis with timer. TO DO

	//urls for images
	URL playUpURL;
	URL stopUpURL;
	URL Up3DURL;
	URL startUpURL;
	URL endUpURL;
	URL rwdUpURL;
	URL ffwUpURL;
	URL playDownURL;
	URL stopDownURL;
	URL Down3DURL;
	URL startDownURL;
	URL endDownURL;
	URL rwdDownURL;
	URL ffwDownURL;

	URL playOverURL;
	URL stopOverURL;
	URL over3DURL;
	URL startOverURL;
	URL endOverURL;
	URL rwdOverURL;
	URL ffwOverURL;
	URL downOver3DURL;

	boolean on3D = false;

	int minA;
	int maxA;
	Hashtable labelTable; //for slider labels

	boolean isInit = false;

	/**
	 * constructor sets up control positions and links to other components 
	 */
	public SlidePanel(
		SliceList list_in,
		TextPanel textPanel_in,
		MainPanel mainPanel_in,
		ColorChooser colorChooser_in,
		StatusBar statusBar_in,
		ImagePanel panel3D_in
	/*,JPanel test3D_in*/
	) {
		super();

		//links
		sliceList = list_in;
		textPanel = textPanel_in;
		mainPanel = mainPanel_in;
		colorChooser = colorChooser_in;
		statusBar = statusBar_in;
		panel3D = panel3D_in;
		colorChooser.setSlider(this);
		mainPanel.setSlider(this);

		int StartList = 0;
		int EndList = 100;

		//Create the Jslider.
		slider = new JSlider(JSlider.HORIZONTAL);

		//create the buttons
		go3D = new JButton();
		play = new JButton();
		stop = new JButton();
		begng = new JButton();
		end = new JButton();
		ffw = new JButton();
		rew = new JButton();

		//set text of buttons
		go3D.setToolTipText("Start 3D Visualisation");
		play.setToolTipText("Play");
		stop.setToolTipText("Stop");
		begng.setToolTipText("Start");
		end.setToolTipText("End");
		ffw.setToolTipText("FFW");
		rew.setToolTipText("REW");

		//load up images
		playUpURL = SlidePanel.class.getResource("resources/playup.gif");
		stopUpURL = SlidePanel.class.getResource("resources/stopbuttonup1.gif");
		Up3DURL = SlidePanel.class.getResource("resources/3dbuttonup.gif");
		startUpURL = SlidePanel.class.getResource("resources/startup.gif");
		endUpURL = SlidePanel.class.getResource("resources/endup.gif");
		rwdUpURL = SlidePanel.class.getResource("resources/rwdup.gif");
		ffwUpURL = SlidePanel.class.getResource("resources/ffwup.gif");

		//load down images
		playDownURL = SlidePanel.class.getResource("resources/playdown.gif");
		stopDownURL = SlidePanel.class.getResource("resources/stopbuttondown1.gif");
		Down3DURL = SlidePanel.class.getResource("resources/3dbuttondown.gif");
		startDownURL = SlidePanel.class.getResource("resources/startdown.gif");
		endDownURL = SlidePanel.class.getResource("resources/enddown.gif");
		rwdDownURL = SlidePanel.class.getResource("resources/rwddown.gif");
		ffwDownURL = SlidePanel.class.getResource("resources/ffwdown.gif");

		//loadover images
		playOverURL = SlidePanel.class.getResource("resources/playover.gif");
		stopOverURL = SlidePanel.class.getResource("resources/stopbuttonover.gif");
		downOver3DURL = SlidePanel.class.getResource("resources/3dbuttondownover.gif");
		over3DURL = SlidePanel.class.getResource("resources/3dbuttonover.gif");
		startOverURL = SlidePanel.class.getResource("resources/startover.gif");
		endOverURL = SlidePanel.class.getResource("resources/endover.gif");
		rwdOverURL = SlidePanel.class.getResource("resources/rwdover.gif");
		ffwOverURL = SlidePanel.class.getResource("resources/ffwover.gif");

		//set up images for buttons
		play.setIcon(new ImageIcon(playUpURL));
		stop.setIcon(new ImageIcon(stopDownURL));
		go3D.setIcon(new ImageIcon(Up3DURL));
		begng.setIcon(new ImageIcon(startUpURL));
		end.setIcon(new ImageIcon(endUpURL));
		ffw.setIcon(new ImageIcon(ffwUpURL));
		rew.setIcon(new ImageIcon(rwdUpURL));

		//set down images for buttons
		play.setPressedIcon(new ImageIcon(playDownURL));
		stop.setPressedIcon(new ImageIcon(stopDownURL));
		go3D.setPressedIcon(new ImageIcon(Down3DURL));
		begng.setPressedIcon(new ImageIcon(startDownURL));
		end.setPressedIcon(new ImageIcon(endDownURL));
		ffw.setPressedIcon(new ImageIcon(ffwDownURL));
		rew.setPressedIcon(new ImageIcon(rwdDownURL));

		//set rollovers
		play.setRolloverEnabled(true);
		stop.setRolloverEnabled(true);
		go3D.setRolloverEnabled(true);
		begng.setRolloverEnabled(true);
		end.setRolloverEnabled(true);
		ffw.setRolloverEnabled(true);
		rew.setRolloverEnabled(true);

		play.setRolloverIcon(new ImageIcon(playOverURL));
		stop.setRolloverIcon(new ImageIcon(stopDownURL));
		go3D.setRolloverIcon(new ImageIcon(over3DURL));
		begng.setRolloverIcon(new ImageIcon(startOverURL));
		end.setRolloverIcon(new ImageIcon(endOverURL));
		ffw.setRolloverIcon(new ImageIcon(ffwOverURL));
		rew.setRolloverIcon(new ImageIcon(rwdOverURL));

		//remove border from buttons
		play.setBorderPainted(false);
		play.setBounds(0, 0, 58, 58);
		play.setSize(58, 58);
		play.setPreferredSize(new Dimension(58, 58));
		play.setBackground(new Color(236, 233, 216));

		stop.setBorderPainted(false);
		stop.setBounds(0, 0, 42, 58);
		stop.setSize(42, 58);
		stop.setPreferredSize(new Dimension(42, 58));
		stop.setBackground(new Color(236, 233, 216));

		go3D.setBorderPainted(false);
		go3D.setBounds(0, 0, 50, 58);
		go3D.setSize(50, 58);
		go3D.setPreferredSize(new Dimension(50, 58));
		go3D.setBackground(new Color(236, 233, 216));

		begng.setBorderPainted(false);
		begng.setBounds(0, 0, 43, 29);
		begng.setSize(43, 29);
		begng.setPreferredSize(new Dimension(43, 29));

		end.setBorderPainted(false);
		end.setBounds(0, 0, 43, 29);
		end.setSize(43, 29);
		end.setPreferredSize(new Dimension(43, 29));

		ffw.setBorderPainted(false);
		ffw.setBounds(0, 0, 43, 29);
		ffw.setSize(43, 29);
		ffw.setPreferredSize(new Dimension(43, 29));

		rew.setBorderPainted(false);
		rew.setBounds(0, 0, 43, 29);
		rew.setSize(43, 29);
		rew.setPreferredSize(new Dimension(43, 29));

		//Create the formatted text field and its formatter.
		numberFormat = java.text.NumberFormat.getIntegerInstance();
		formatter = new NumberFormatter(numberFormat);

		textField = new JFormattedTextField(formatter);
		textField.setBackground(Color.LIGHT_GRAY);
		textField.setColumns(2); //get some space

		//Create the mouselistener
		slider.addMouseListener(this);
		addMouseListener(this);

		//Lay out the content.
		setPreferredSize(new Dimension(510, 70));

		//set up button packs
		JPanel startRew = new JPanel();
		startRew.add(rew, BorderLayout.NORTH);
		startRew.add(begng, BorderLayout.SOUTH);
		startRew.setPreferredSize(new Dimension(45, 70));

		JPanel sliderAndText = new JPanel();
		sliderAndText.add(slider, BorderLayout.WEST);
		//	sliderAndText.add(textField,BorderLayout.EAST);
		sliderAndText.setPreferredSize(new Dimension(200, 70));

		JPanel endFfw = new JPanel();
		endFfw.add(ffw, BorderLayout.NORTH);
		endFfw.add(end, BorderLayout.SOUTH);
		endFfw.setPreferredSize(new Dimension(45, 70));

		JPanel playStop = new JPanel();
		playStop.add(play, BorderLayout.WEST);
		playStop.add(stop, BorderLayout.CENTER);
		playStop.add(go3D, BorderLayout.EAST); //add 3d button here
		playStop.setPreferredSize(new Dimension(165, 70));

		setLayout(new FlowLayout());
		add(startRew);
		add(sliderAndText);
		add(endFfw);
		add(playStop);

		//set up background colors for the button panels
		startRew.setBackground(new Color(236, 233, 216));
		sliderAndText.setBackground(new Color(236, 233, 216));
		endFfw.setBackground(new Color(236, 233, 216));
		playStop.setBackground(new Color(236, 233, 216));
		slider.setBackground(new Color(236, 233, 216));

		//Add border around the select panel.
		setBorder(
			BorderFactory.createCompoundBorder(
				BorderFactory.createTitledBorder("TimeStep Controls: "),
				BorderFactory.createEmptyBorder(0, 0, 0, 0)));
	}

	/**
	 * initialises controls (button press listeners and the action listener for the slider
	 * also sets up the range of the slider and the labels of it
	 * @param min - minimum timestep
	 * @param max - maximum timestep
	 */
	public void initControls(int min, int max) {

		//reset controls and timers
		resetButtons();
		if (playTimer != null) {
			playTimer.cancel();
		}
		stopDown();

		//set up slider
		minA = min;
		maxA = max;

		//first remove all the action listeners fromthe slider if used
		if (isInit != false) {
			slider.removeChangeListener(slider.getChangeListeners()[0]);
			go3D.removeActionListener(go3D.getActionListeners()[0]);
			play.removeActionListener(play.getActionListeners()[0]);
			stop.removeActionListener(stop.getActionListeners()[0]);
			begng.removeActionListener(begng.getActionListeners()[0]);
			end.removeActionListener(end.getActionListeners()[0]);
			ffw.removeActionListener(ffw.getActionListeners()[0]);
			rew.removeActionListener(rew.getActionListeners()[0]);
		}

		//set up formatting
		slider.setToolTipText("Timeslice selection");
		slider.putClientProperty("JSlider.isFilled", Boolean.TRUE);
		slider.setSnapToTicks(false);
		slider.setMinimum(min);
		slider.setMaximum(max);

		slider.setPaintTicks(false);
		slider.setPaintLabels(false);

		//set up labels
		if (max < 5) {
			slider.createStandardLabels(1);
			slider.setMajorTickSpacing(1);
		}
		if (max >= 5) {
			int maxInt = (int) max / 5;
			slider.createStandardLabels(maxInt);

			//	Create the label table
			labelTable = null;
			labelTable = new Hashtable();
			labelTable.put(new Integer(0), new JLabel("0"));
			labelTable.put(new Integer(maxInt), new JLabel("" + maxInt));
			labelTable.put(
				new Integer(maxInt * 2),
				new JLabel("" + (maxInt * 2)));
			labelTable.put(
				new Integer(maxInt * 3),
				new JLabel("" + maxInt * 3));
			labelTable.put(
				new Integer(maxInt * 4),
				new JLabel("" + (maxInt * 4)));
			labelTable.put(
				new Integer(maxInt * 5),
				new JLabel("" + (maxInt * 5)));
			slider.setLabelTable(labelTable);
			slider.setMajorTickSpacing((int) max / 5);
		}

		slider.setPaintTicks(true);
		slider.setPaintLabels(true);

		//set slider postion at start of file
		slider.setValue(0);

		//set 3d flag to false
		on3D = false;

		//add event handlers for slider change
		slider.addChangeListener(new ChangeListener() {
			public void stateChanged(ChangeEvent evt) {
				printOut(evt);
				JSlider source = (JSlider) evt.getSource();
				int ts = (int) source.getValue();
				if (!source.getValueIsAdjusting()) { //done adjusting
					textField.setValue(new Integer(ts));
					statusBar.setStatus(
						"Current Timestep = "
							+ slider.getValue()
							+ "     Min: "
							+ minA
							+ "   Max: "
							+ maxA);
					//update ts value		
				} else { //value is adjusting; just set the text
					textField.setText(String.valueOf(ts));
					statusBar.setStatus(
						"Current Timestep = "
							+ String.valueOf(ts)
							+ "     Min: "
							+ minA
							+ "   Max: "
							+ maxA);
				}

			}
		});

		//text field - NOT USED
		formatter.setMinimum(new Integer(min));
		formatter.setMaximum(new Integer(max));
		textField.setValue(new Integer(0));
		//		React when the user presses Enter.
		textField.getInputMap().put(
			KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0),
			"check");
		textField.getActionMap().put("check", new AbstractAction() {
			public void actionPerformed(ActionEvent e) {
				if (!textField.isEditValid()) { //The text is invalid.
					Toolkit.getDefaultToolkit().beep();
					textField.selectAll();
				} else
					try { //The text is valid,
						textField.commitEdit(); //so use it.
					} catch (java.text.ParseException exc) {
					}
			}
		});

		textField.addPropertyChangeListener(this);

		//add button listerners
		go3D.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {

				//case if the 3d visualisation is NOT initialised
				if (on3D == false) {
					resetButtons();
					stopDown();
					down3D();
					if (playTimer != null) {
						playTimer.cancel();
					}
					try {
						initTevi(evt);
						on3D = true;

					} catch (IOException ioe) {
					}
				} else {
					//case if the 3d visualisation IS initialised
					//STOP the 3D visualisation, and replace splash.
					panel3D.removeAll();
					tevi.destroy();
					tevi = null;
					statusBar.setStatus("3D Visualisation stopped");
					System.gc();
					URL splash3DURL =
						MainPanel.class.getResource("3dsplash.jpg");
					Toolkit toolkit = Toolkit.getDefaultToolkit();
					panel3D.SetImage(toolkit.getImage(splash3DURL));
					//reset3dButton();
					go3D.setIcon(new ImageIcon(Up3DURL));
					go3D.setRolloverIcon(new ImageIcon(over3DURL));
					go3D.setToolTipText("Start 3D Visualisation");

					resetButtons();
					if (playTimer != null) {
						playTimer.cancel();
					}
					stopDown();
					on3D = false;

				}
			}
		});

		play.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				resetButtons();
				playDown();
				if (playTimer != null) {
					playTimer.cancel();
				}
				playTimer = new Timer();
				playTimer.schedule(new PlayTask(1), 0, //initial delay
				1 * 1000); //subsequent rate

			}
		});

		stop.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				resetButtons();
				stopDown();
				if (playTimer != null) {
					playTimer.cancel();
				}

			}
		});

		begng.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				//	resetButtons();
				slider.setValue(slider.getMinimum());
				//	stopDown();
			}
		});

		end.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				//	resetButtons();
				slider.setValue(slider.getMaximum());
				//	stopDown();
			}
		});

		ffw.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				resetButtons();
				ffwDown();
				if (playTimer != null) {
					playTimer.cancel();
				}
				playTimer = new Timer();
				playTimer.schedule(new PlayTask(5), 0, //initial delay
				1 * 500); //subsequent rate

			}
		});

		rew.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				resetButtons();
				rewDown();
				if (playTimer != null) {
					playTimer.cancel();
				}
				playTimer = new Timer();
				playTimer.schedule(new PlayTask(-5), 0, //initial delay
				1 * 500); //subsequent rate
			}
		});

		//if 3d already on
		if (tevi != null) {
			panel3D.removeAll();
			tevi.destroy();
			tevi = null;
			statusBar.setStatus("3D Visualisation stopped");
			System.gc();
			URL splash3DURL = MainPanel.class.getResource("3dsplash.jpg");
			Toolkit toolkit = Toolkit.getDefaultToolkit();
			panel3D.SetImage(toolkit.getImage(splash3DURL));
			//reset3dButton();
			go3D.setIcon(new ImageIcon(Up3DURL));
			go3D.setRolloverIcon(new ImageIcon(over3DURL));
			go3D.setToolTipText("Start 3D Visualisation");
			stopDown();
			on3D = false;
		}

		//draw first slice to 2D screens
		printOut();

		//draw fitness key
		displayFKey();

		//set initialised flag
		isInit = true;
	}

	/**
	 * timer thread classes which deal with the buttons and slide 
	 * increment
	 */
	class PlayTask extends TimerTask {
		int amount = 0;

		public PlayTask(int amount_in) {
			amount = amount_in;
		}

		public void run() {
			//effort to synchronise updates - will do later
//			if (on3D==true){
//				while(changed==false){
//					try {
//						wait();
//					} catch (InterruptedException e) {
//						e.printStackTrace();
//					}
//				}
//			}
			//for going forwards
			if (amount > 0) {
				if (slider.getValue() < slider.getMaximum()) {
					slider.setValue(slider.getValue() + amount);
					changed = false;
				} else {
					resetButtons();
					stopDown();
					playTimer.cancel();
				}
			}
			//for backwards
			if (amount < 0) {
				if (slider.getValue() > slider.getMinimum()) {
					slider.setValue(slider.getValue() + amount);
					changed = false;
				} else {
					resetButtons();
					stopDown();
					playTimer.cancel();
				}
			}

		}
//		public void setChanged(boolean drawn){
//				changed = drawn;
//				notifyAll();
//			}
	}

	

	/**
	 * the following methods reset all the buttons to up
	 * 
	 */
	private void resetButtons() {
		play.setIcon(new ImageIcon(playUpURL));
		stop.setIcon(new ImageIcon(stopUpURL));
		begng.setIcon(new ImageIcon(startUpURL));
		end.setIcon(new ImageIcon(endUpURL));
		ffw.setIcon(new ImageIcon(ffwUpURL));
		rew.setIcon(new ImageIcon(rwdUpURL));

		play.setRolloverIcon(new ImageIcon(playOverURL));
		stop.setRolloverIcon(new ImageIcon(stopOverURL));
		begng.setRolloverIcon(new ImageIcon(startOverURL));
		end.setRolloverIcon(new ImageIcon(endOverURL));
		ffw.setRolloverIcon(new ImageIcon(ffwOverURL));
		rew.setRolloverIcon(new ImageIcon(rwdOverURL));
	}

	/**
	 * the following methods set the respective buttons to appear depressed
	 */
	private void playDown() {
		play.setRolloverIcon(new ImageIcon(playDownURL));
		play.setIcon(new ImageIcon(playDownURL));
	}
	private void stopDown() {
		stop.setRolloverIcon(new ImageIcon(stopDownURL));
		stop.setIcon(new ImageIcon(stopDownURL));
	}
	private void begngDown() {
		begng.setRolloverIcon(new ImageIcon(startDownURL));
		begng.setIcon(new ImageIcon(startDownURL));
	}
	private void endDown() {
		end.setRolloverIcon(new ImageIcon(endDownURL));
		end.setIcon(new ImageIcon(endDownURL));
	}
	private void ffwDown() {
		ffw.setRolloverIcon(new ImageIcon(ffwDownURL));
		ffw.setIcon(new ImageIcon(ffwDownURL));
	}
	private void rewDown() {
		rew.setRolloverIcon(new ImageIcon(rwdDownURL));
		rew.setIcon(new ImageIcon(rwdDownURL));
	}
	private void down3D() {
		go3D.setRolloverIcon(new ImageIcon(downOver3DURL));
		go3D.setIcon(new ImageIcon(Down3DURL));
		go3D.setToolTipText("Stop Visualisation");
	}

	/**
	* Listen to the text field.  This method detects when the
	* value of the text field changes. - NOT USED
	*/
	public void propertyChange(PropertyChangeEvent e) {
		if ("value".equals(e.getPropertyName())) {
			Number value = (Number) e.getNewValue();
			if (slider != null && value != null) {
				slider.setValue(value.intValue());
			}
		}
	}

	private void displayFKey() {
		mainPanel.setKeyF(colorChooser.getFKey());
	}

	/**
	 * print out method - The start of the render pipeline
	 * creates and sets relevant images, and also updates 3D scene if activated
	 * @param evt
	 */
	private void printOut(ChangeEvent evt) {
		textPanel.clearPanel();
		sliceList.printSlice(slider.getValue());
		textPanel.toTop();
		int[] height = sliceList.getSlice(slider.getValue()).getHeightMap();
		int[] freq = sliceList.getSlice(slider.getValue()).getColorMap();
		int sizeX = sliceList.getSlice(slider.getValue()).getArraySizeX();
		int sizeY = sliceList.getSlice(slider.getValue()).getArraySizeY();

		//System.out.println("1 D array X = " + sizeX + " Y = " + sizeY);
		height2D =
			createImage(new MemoryImageSource(sizeX, sizeY, height, 0, sizeX));
		color2D =
			createImage(new MemoryImageSource(sizeX, sizeY, freq, 0, sizeX));
		mainPanel.setImage(height2D, color2D);
		//slider.requestFocusInWindow(); //formerly textPanel

		imageAvailiable = true;

		if (tevi != null) {
			tevi.terrainChange(
				height2D.getScaledInstance(129, 129, Image.SCALE_SMOOTH),
				color2D.getScaledInstance(256, 256, Image.SCALE_SMOOTH));
		}

	}

	/**
	 * Dupilcate print out method w/o event- The start of the render pipeline
	 * creates and sets relevant images, and also updates 3D scene if activated
	 */
	public void printOut() {
		textPanel.clearPanel();
		sliceList.printSlice(slider.getValue());
		textPanel.toTop();
		int[] height = sliceList.getSlice(slider.getValue()).getHeightMap();
		int[] freq = sliceList.getSlice(slider.getValue()).getColorMap();
		int sizeX = sliceList.getSlice(slider.getValue()).getArraySizeX();
		int sizeY = sliceList.getSlice(slider.getValue()).getArraySizeY();

		//System.out.println("1 D array X = " + sizeX + " Y = " + sizeY);
		height2D =
			createImage(new MemoryImageSource(sizeX, sizeY, height, 0, sizeX));
		color2D =
			createImage(new MemoryImageSource(sizeX, sizeY, freq, 0, sizeX));
		mainPanel.setImage(height2D, color2D);
		//slider.requestFocusInWindow(); //formerly textPanel

		imageAvailiable = true;

		if (tevi != null) {
			tevi.terrainChange(
				height2D.getScaledInstance(129, 129, Image.SCALE_SMOOTH),
				color2D.getScaledInstance(256, 256, Image.SCALE_SMOOTH));
		}

	}

	/**
	 * set up links with 3d visualisation
	 */
	public void setTevi(TeVi tevi_in) {
		tevi = tevi_in;
	}

	/**
	 * initialise 3d visualisation
	 */
	private void initTevi(ActionEvent evt) throws IOException {
		/*try for tevi init*/
		if (imageAvailiable == true) {

			Image nullImage = null;
			panel3D.SetImage(nullImage);
			panel3D.removeAll();
			System.gc();

			TeVi newTevi =
				new TeVi(
					height2D.getScaledInstance(129, 129, Image.SCALE_SMOOTH),
					color2D.getScaledInstance(256, 256, Image.SCALE_SMOOTH),
					this);

			tevi.setPreferredSize(new Dimension(440, 445));
			panel3D.add(tevi);

			slider.setValue(slider.getValue());//force refresh to display
			statusBar.setStatus("3D Visualisation Initialised");
		}
	}

	/**
	 * These methods implement the mouselistener interface. 
	 * They are only here to set up the focus on mouse clicks...
	 */
	public void mouseClicked(MouseEvent arg0) {
		//slider.requestFocusInWindow(); //formerly textPanel
	}

	/* (non-Javadoc)
	 * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
	 */
	public void mousePressed(MouseEvent arg0) {
		//slider.requestFocusInWindow(); //formerly textPanel

	}

	/* (non-Javadoc)
	 * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
	 */
	public void mouseReleased(MouseEvent arg0) {
		// TODO Auto-generated method stub

	}

	/* (non-Javadoc)
	 * @see java.awt.event.MouseListener#mouseEntered(java.awt.event.MouseEvent)
	 */
	public void mouseEntered(MouseEvent arg0) {
		// TODO Auto-generated method stub

	}

	/* (non-Javadoc)
	 * @see java.awt.event.MouseListener#mouseExited(java.awt.event.MouseEvent)
	 */
	public void mouseExited(MouseEvent arg0) {
		// TODO Auto-generated method stub

	}

	/* (non-Javadoc)
	 * @see javax.swing.event.ChangeListener#stateChanged(javax.swing.event.ChangeEvent)
	 */
	public void stateChanged(ChangeEvent e) {
		// TODO Auto-generated method stub

	}

	/* (non-Javadoc)
	 * @see java.awt.event.WindowListener#windowActivated(java.awt.event.WindowEvent)
	 */
	public void windowActivated(WindowEvent e) {
		// TODO Auto-generated method stub

	}

	/* (non-Javadoc)
	 * @see java.awt.event.WindowListener#windowClosed(java.awt.event.WindowEvent)
	 */
	public void windowClosed(WindowEvent e) {
		// TODO Auto-generated method stub
		System.out.println("got this far");
	}

	/* (non-Javadoc)
	 * @see java.awt.event.WindowListener#windowClosing(java.awt.event.WindowEvent)
	 */
	public void windowClosing(WindowEvent e) {
		// TODO Auto-generated method stub
		System.out.println("got this far");
		frame.dispose();
		frame = null;
	}

	/* (non-Javadoc)
	 * @see java.awt.event.WindowListener#windowDeactivated(java.awt.event.WindowEvent)
	 */
	public void windowDeactivated(WindowEvent e) {
		// TODO Auto-generated method stub

	}

	/* (non-Javadoc)
	 * @see java.awt.event.WindowListener#windowDeiconified(java.awt.event.WindowEvent)
	 */
	public void windowDeiconified(WindowEvent e) {
		// TODO Auto-generated method stub

	}

	/* (non-Javadoc)
	 * @see java.awt.event.WindowListener#windowIconified(java.awt.event.WindowEvent)
	 */
	public void windowIconified(WindowEvent e) {
		// TODO Auto-generated method stub

	}

	/* (non-Javadoc)
	 * @see java.awt.event.WindowListener#windowOpened(java.awt.event.WindowEvent)
	 */
	public void windowOpened(WindowEvent e) {
		// TODO Auto-generated method stub

	}

	public void paintComponent(Graphics g) {

		//Make background grey
		g.setColor(new Color(236, 233, 216));
		g.fillRect(2, 17, getSize().width - 6, getSize().height - 20);
		g.setColor(Color.black);
	}

}
