//EPDWorld.m EPD
//Copyright James Marshall 1998-2004. Freely distributable under the GNU General Public Licence

#import "EPDWorld.h"
#import "EPDAgent.h"

@implementation EPDWorld

-setupMaxAgents: (int) maxPop maxLocalAgents: (int) maxLocalPop startAgents: (int) startPop strategyLength: (int) stratLen strategyNumber: (int) stratNum deathProbability: (double) deathProb moveProbability: (double) moveProb defectorProbability: (double) defectorProb mutationRate: (float) mu crossProbability: (float) crossProb agentList: aListIndex payoffMatrix: anArray worldDisplay: aDisplay populationDisplay: bDisplay startEnergy: (int) energy livingCost: (int) cost
{
  //initialise the world parameters and objects
  int l;
  id tempList;
  worldTime=0;
  maxAgents=maxPop;
  maxLocalAgents=maxLocalPop;
  startAgents=startPop;
  strategyLength=stratLen;
  strategyNumber=stratNum;
  deathProbability=deathProb;
  moveProbability=moveProb;
  defectorProbability=defectorProb;
  mutationRate=mu;
  crossProbability=crossProb;
  startEnergy=energy;
  livingCost=cost;
  allAgents=aListIndex;
  payoffs=anArray;
  worldDisplay=aDisplay;
  populationDisplay=bDisplay;
  totalActions=0;
  cooperationLevel=0;
  totalInteractions=0;
  totalInteractionLength=0;
  totalPartners=0;
  totalInteractants=0;
  totalPopulationSize=0;
  totalPopulatedCells=0;
  //create the allele frequency array
  alleleFrequencies=[Array create: [self getZone] setCount: strategyLength];
  for (l=0; l<strategyLength; l++)
  {
    [alleleFrequencies atOffset: l put: [IntegerData create: [self getZone]]];
  }
  alleleTotals=[Array create: [self getZone] setCount: strategyLength];
  for (l=0; l<strategyLength; l++)
  {
    [alleleTotals atOffset: l put: [IntegerData create: [self getZone]]];
  }
  strategyFrequencies=[Array create: [self getZone] setCount: strategyNumber];
  for (l=0; l<strategyNumber; l++)
  {
    [strategyFrequencies atOffset: l put: [IntegerData create: [self getZone]]];
  }
  //create the cooperation and relatedness level histories
  tempList=[List create: [self getZone]];
  cooperationValues=[tempList begin: [tempList getZone]];
  tempList=[List create: [self getZone]];
  relatednessValues=[tempList begin: [tempList getZone]];

  //create the lattice to store the total diversity of cells
  cellTotalDiversity=[Discrete2d createBegin: globalZone];
  [cellTotalDiversity setSizeX: [self getSizeX] Y: [self getSizeY]];
  cellTotalDiversity=[cellTotalDiversity createEnd];
  [cellTotalDiversity fillWithValue: 0];
  //create the lattice to store the total diversity of cells
  cellDiversityDivisor=[Discrete2d createBegin: globalZone];
  [cellDiversityDivisor setSizeX: [self getSizeX] Y: [self getSizeY]];
  cellDiversityDivisor=[cellDiversityDivisor createEnd];
  [cellDiversityDivisor fillWithValue: 0];
  //create the lattice to store the total diversity of cells
  cellTotalCooperativeActions=[Discrete2d createBegin: globalZone];
  [cellTotalCooperativeActions setSizeX: [self getSizeX] Y: [self getSizeY]];
  cellTotalCooperativeActions=[cellTotalCooperativeActions createEnd];
  [cellTotalCooperativeActions fillWithValue: 0];
  //create the lattice to store the total diversity of cells
  cellTotalActions=[Discrete2d createBegin: globalZone];
  [cellTotalActions setSizeX: [self getSizeX] Y: [self getSizeY]];
  cellTotalActions=[cellTotalActions createEnd];
  [cellTotalActions fillWithValue: 0];

  return self;
}

-createEnd
{
  return [super createEnd];
}

-(void)drop
{
  int l,x,y;

  for (l=0; l<strategyLength; l++)
  {
    [[alleleFrequencies atOffset: l] drop];
    [[alleleTotals atOffset: l] drop];    
    [[strategyFrequencies atOffset: l] drop];    
  }
  //drop the allele frequency array
  [alleleFrequencies drop];
  //drop the allele totals array
  [alleleTotals drop];
  //drop the strategy frequency array
  [strategyFrequencies drop];
  //drop the cooperation and relatedness level histories
  [[cooperationValues getCollection] deleteAll];
  [[cooperationValues getCollection] drop];
  [cooperationValues drop];
  [[relatednessValues getCollection] deleteAll];
  [[relatednessValues getCollection] drop];
  [relatednessValues drop];
  //drop the local cell lists
  for (x=0; x<xsize; x++)
  {
    for (y=0; y<ysize; y++)
    {
      [[[self getObjectAtX: x Y: y] getCollection] drop];
      [[self getObjectAtX: x Y: y] drop];
    }
  }
  [cellTotalDiversity drop];
  [cellDiversityDivisor drop];
  [cellTotalCooperativeActions drop];
  [cellTotalActions drop];

  [super drop];
}

-populate
{
  int l1,l2,x,y,stratNum;
  id newAgent;
  id <List> tempList;
//  id <ListIndex> localList;
  id localList;

  //create the local cell lists
  for (x=0; x<xsize; x++)
  {
    for (y=0; y<ysize; y++)
    {
      tempList=[List create: [self getZone]];
      localList=[tempList begin: [tempList getZone]];
      [self putObject: localList atX: x Y: y];
    }
  }
  //create the initial agent population
  for (l1=0; l1<startAgents; l1++)
  {
    //randomly place agent in environment
    x=[uniformIntRand getIntegerWithMin: 0 withMax: xsize-1];
    y=[uniformIntRand getIntegerWithMin: 0 withMax: ysize-1];
    //create new agent
    newAgent=[EPDAgent createBegin: [self getZone]];
    [newAgent setupWorld: self x: x y: y];
    newAgent=[newAgent createEnd];
    //initialise agent's strategy chromosome with given defector probability
    for (l2=0; l2<strategyLength; l2++)
    {
      if ([uniformDblRand getDoubleWithMin: 0 withMax: 1]>defectorProbability || defectorProbability==0)
      {
        [newAgent setStrategyAtOffset: l2 value: 1];
      }
      else
      {
        [newAgent setStrategyAtOffset: l2 value: 0];
      }
//      [newAgent setStrategyAtOffset: l2 value: [uniformIntRand getIntegerWithMin: 0 withMax: 1]];
      if ((int)[[newAgent getStrategy] atOffset: l2]==1)
      {
        [self increaseAlleleFrequency: l2];
      }
    }
    stratNum=(int)[[newAgent getStrategy] atOffset: 0];
    stratNum+=(int)[[newAgent getStrategy] atOffset: 1]*2;
    stratNum+=(int)[[newAgent getStrategy] atOffset: 2]*4;
    stratNum+=(int)[[newAgent getStrategy] atOffset: 3]*8;
    stratNum+=(int)[[newAgent getStrategy] atOffset: 4]*16;

    /*  replaced with above, S. TOKUMINE 
    stratNum=(int)[[newAgent getStrategy] atOffset: 2];
    stratNum+=(int)[[newAgent getStrategy] atOffset: 3]*2;
    stratNum+=(int)[[newAgent getStrategy] atOffset: 0]*4;
    */  
    
    [self increaseStrategyFrequency: stratNum];

    [[[self getObjectAtX: x Y: y] getCollection] addFirst: newAgent];
    [[allAgents getCollection] addFirst: newAgent];
  }

  return self;
}

-addAgent: newAgent
{
  //add a new agent to the global list
  [[allAgents getCollection] addLast: newAgent];

  return self;
}

-purge
{
  id dyingAgent, forgettingAgent, historyItem;

  //remove agents who have died this step from the environment
  [allAgents setLoc: Start];
  while ((dyingAgent=[allAgents next]))
  {
    if ([dyingAgent isAlive]==NO) //check if this agent is dead
    {
      //store agent's interaction length statistics
      totalInteractants++;
      [[dyingAgent getHistory] setLoc: Start];
      while ((historyItem=[[dyingAgent getHistory] next]))
      {
        totalInteractions++;
        totalInteractionLength+=(int)[historyItem atOffset: 3];
        totalPartners++;
      }

      //remove histories involving the agent from the other agents' memories
      [allAgents setLoc: Start];
      while ((forgettingAgent=[allAgents next]))
      {
        [[forgettingAgent getHistory] setLoc: Start];
        while ((historyItem=[[forgettingAgent getHistory] next]))
        {
          if ([historyItem atOffset: 0]==dyingAgent)
          {
            [[forgettingAgent getHistory] remove];
            [historyItem drop];
          }
        }
      }

      [allAgents setLoc: Start];
      [allAgents findNext: dyingAgent];
      //remove the agent from the global list
      dyingAgent=[allAgents remove];
      [dyingAgent drop];
    }
  }

  return self;
}

-mate
{
  //randomise order of mating opportunities in global agent list
  int l1, l2, population, agentNumber, numAgentsMated;
  id agent, agentsMated;

  population=[self getWorldPopulation];
  if (population>0)
  {
    //create temporary array to remember if agents have had their mating opportunity (represented by '1')
    agentsMated=[Array create: [self getZone] setCount: population];
    for (l1=0; l1<population; l1++)
    {
      [agentsMated atOffset: l1 put:(id)0];
    }

    numAgentsMated=0;
    for (l1=0; l1<population; l1++) //repeat for every agent in the population
    {
      //randomly choose an agent which hasn't had its mating opportunity
/* the old inefficient way of choosing
      agentNumber=[uniformIntRand getIntegerWithMin: 0 withMax: population-1];
      while ((int)[agentsMated atOffset: agentNumber]==1)
      {
        agentNumber=[uniformIntRand getIntegerWithMin: 0 withMax: population-1];
      }
      [agentsMated atOffset: agentNumber put:(id)1]; //flag the agent as having had its mating opportunity
      //retrieve the agent from the global list
      [allAgents setLoc: Start];
      agent=[allAgents get]; // here to prevent compiler warning
      for (l2=0; l2<=agentNumber; l2++)
      {
        agent=[allAgents next];
      }

      [agent mate];
*/
      if (population-numAgentsMated>0)
      {
        agentNumber=[uniformIntRand getIntegerWithMin: 0 withMax: (population-numAgentsMated)-1];
        [allAgents setLoc: Start];
        agent=[allAgents get]; // here to prevent compiler warning
        l2=0;
        while (agentNumber>=0)
        {
          agent=[allAgents next];
          if ((int)[agentsMated atOffset: l2]!=1)
          {
            agentNumber--;
          }
          if (agentNumber>=0)
          {
            l2++;
          }
        }
        [agentsMated atOffset: l2 put:(id)1]; //flag the agent as having had its mating opportunity
        [agent mate]; //allow the agent to mate
        numAgentsMated++;
      }
    }
    [agentsMated drop];
  }

  return self;
}

-display
{
  id alleles;
  int x, y, l1, localPopulation;
#ifndef relatedness_measure3
  id agent;
#endif
#ifdef relatedness_measure2
  int populatedCells;
#endif
  double diversity;
  localPopulation=0;
  //create a temporary array used in calculating relatedness within a cell
  alleles=[Array create: [self getZone] setCount: strategyLength];
  for (l1=0; l1<strategyLength; l1++)
  {
    [alleles atOffset: l1 put: [DoubleData create: [self getZone]]];
  }

  #ifdef relatedness_measure2
    //calculate local cooperation and relatedness levels for all cells inhabited by agents
    populatedCells=0;
    globalRelatedness=0;
  #endif

  #ifdef relatedness_measure3
    for (x=0; x<xsize; x++)
    {
      for (y=0; y<ysize; y++)
      {
        if ([[[self getObjectAtX: x Y: y] getCollection] getCount]>0) //check if cell is inhabited
        {
          localPopulation=[[[self getObjectAtX: x Y: y] getCollection] getCount];
          if ([[[self getObjectAtX: x Y: y] getCollection] getCount]==1)
          {
            //cell inhabited by only one agent, colour red
            [worldDisplay putValue: 122 atX: x Y: y];
          }
          else
          {
            diversity=[self getDiversityLevelAtX: x Y: y];
             //draw the cooperation and relatedness of this cell on the world display
            [worldDisplay putValue: (int)([self getCooperationLevelAtX: x Y: y]*10)*11+(1-diversity)*10+1 atX: x Y: y];
          }
          //draw the population density of this cell on the population display
          [populationDisplay putValue: (long)(((double)localPopulation/maxAgents)*10+1) atX: x Y: y];
        }
        else
        {
          //draw this cell as uninhabited on the population and world displays
          [populationDisplay putValue: 0 atX: x Y: y];
          [worldDisplay putValue: 0 atX: x Y: y];
        }
      }
    }
  #else
    for (x=0; x<xsize; x++)
    {
      for (y=0; y<ysize; y++)
      {
        if ([[[self getObjectAtX: x Y: y] getCollection] getCount]>0) //check if cell is inhabited
        {
          localPopulation=0;
          [[self getObjectAtX: x Y: y] setLoc: Start];
          //count the frequency of each allele in all the agents in this cell, and the percentage of cooperative actions
          for (l1=0; l1<strategyLength; l1++)
          {
            [[alleles atOffset: l1] setDoubleData: 0];
          }
          while ((agent=[[self getObjectAtX: x Y: y] next]))
          {
            localPopulation++;
            for (l1=0; l1<strategyLength; l1++)
            {
              [[alleles atOffset: l1] setDoubleData: [[alleles atOffset: l1] getDoubleData]+(int)[[agent getStrategy] atOffset: l1]];
            }
          }
          //calculate the genetic diversity of the population in this cell
          diversity=0;
          for (l1=0; l1<strategyLength; l1++)
          {
            [[alleles atOffset: l1] setDoubleData: [[alleles atOffset: l1] getDoubleData]/[[[self getObjectAtX: x Y: y] getCollection] getCount]];
            diversity=diversity+1-(4*((0.5-[[alleles atOffset: l1] getDoubleData])*(0.5-[[alleles atOffset: l1] getDoubleData])));
          }
          diversity=diversity/strategyLength;
          //draw the population density of this cell on the population display
          [populationDisplay putValue: (long)(((double)localPopulation/maxAgents)*10+1) atX: x Y: y];
          //draw the cooperation and relatedness levels of this cell on the world display
          if (localPopulation==1)
          {
            //cell inhabited by only one agent, colour red
            [worldDisplay putValue: 122 atX: x Y: y];
          }
          else
          {
            //draw the cooperation and relatedness of this cell on the world display
            [worldDisplay putValue: (int)([self getCooperationLevelAtX: x Y: y]*10)*11+(1-diversity)*10+1 atX: x Y: y];

            #ifdef relatedness_measure2
              //increment counter of number of cells inhabited by more than one agent
              populatedCells++;
              //add cell relatedness to global relatedness total
              globalRelatedness+=(1-diversity);
            #endif
          }
        }
        else
        {
          //draw this cell as uninhabited on the population and world displays
          [populationDisplay putValue: 0 atX: x Y: y];
          [worldDisplay putValue: 0 atX: x Y: y];
        }
      }
    }
  #endif

  #ifdef testcard
    for (x=0; x<11; x++)
    {
      for (y=0; y<11; y++)
      {
        [worldDisplay putValue: (int)((x*11)+y+1) atX: x Y: y];
      }
    }
  #endif

  //drop the allele frequency array
  for (l1=1; l1<strategyLength; l1++)
  {
    [[alleles atOffset: l1] drop];
  }
  [alleles drop];

  #ifdef relatedness_measure2
    //calculate global relatedness
    if (populatedCells>0)
    {
      globalRelatedness=globalRelatedness/(double)populatedCells;
      globalRelatedness=globalRelatedness*100;
    }
    else
    {
      globalRelatedness=0;
    }
  #endif

  return self;
}

-(int)getMaxAgents
{
  //return the maximum number of agents the world can support
  return maxAgents;
}

-(int)getMaxLocalAgents
{
  //return the maximum number of agents a single cell can support
  return maxLocalAgents;
}

-(int)getWorldPopulation
{
  //return the current world population
  return (int)[[allAgents getCollection] getCount];
}

-getAllAgents
{
  //return the world population list
  return allAgents;
}

-(int)getGlobalRelatedness
{
  //return the global relatedness of the agent population
  #ifdef relatedness_measure1
    id alleles, agent;
    int l1;
    double diversity;

    if ([self getWorldPopulation]>0)
    {
      //count frequencies of alleles in all agents in the world
      alleles=[Array create: [self getZone] setCount: strategyLength];
      for (l1=0; l1<strategyLength; l1++)
      {
        [alleles atOffset: l1 put: [DoubleData create: [self getZone]]];
        [[alleles atOffset: l1] setDoubleData: 0];
      }
      [allAgents setLoc: Start];
      while ((agent=[allAgents next]))
      {
        for (l1=0; l1<strategyLength; l1++)
        {
          [[alleles atOffset: l1] setDoubleData: [[alleles atOffset: l1] getDoubleData]+(int)[[agent getStrategy] atOffset: l1]];
        }
      }
      //calculate genetic diversity of all agents in the world
      diversity=0;
      for (l1=0; l1<strategyLength; l1++)
      {
        [[alleles atOffset: l1] setDoubleData: [[alleles atOffset: l1] getDoubleData]/[[allAgents getCollection] getCount]];
        diversity=diversity+1-(4*((0.5-[[alleles atOffset: l1] getDoubleData])*(0.5-[[alleles atOffset: l1] getDoubleData])));
      }
      diversity=diversity/strategyLength;
      globalRelatedness=(1-diversity)*100;
      //drop allele frequency array
      for (l1=0; l1<strategyLength; l1++)
      {
        [[alleles atOffset: l1] drop];
      }
      [alleles drop];
    }
    else
    {
      globalRelatedness=0;
    }
    return (int)globalRelatedness;
  #endif

  #ifdef relatedness_measure2
    return (int)globalRelatedness;
  #endif

  #ifdef relatedness_measure3
    if (totalActions>0)
    {
      return (int)((globalRelatedness/(double)(totalActions/2))*100);
    }
    else
    {
      return 0;
    }
  #endif
}

-increaseGlobalRelatedness: (double) increase;
{
  globalRelatedness+=increase;

  return self;
}

-resetGlobalRelatedness
{
  globalRelatedness=0;

  return self;
}

-(int)getStrategyLength
{
  //return the length of agents' strategy chromosomes
  return strategyLength;
}

-(double)getDeathProbability
{
  //return the envorinmental probability of death
  return deathProbability;
}

-(double)getMoveProbability
{
  //return the agent movement probability
  return moveProbability;
}

-(float)getMutationRate
{
  //return the mutation rate
  return mutationRate;
}

-(float)getCrossProbability
{
  //return the parameterised crossover probability
  return crossProbability;
}

-(int)getPayoffAction1: (int) action1 action2: (int) action2;
{
  //return the PD payoff for an action pair
  return (int)[[payoffs atOffset: action1] atOffset: action2];
}

-resetCoopLevel;
{
  //reset the variables used in calculating the global cooperation level
  totalActions=0;
  cooperativeActions=0;

  return self;
}

-(int)getCoopLevel
{
  //return the global cooperation level
  if (totalActions>0)
  {
    cooperationLevel=(double)cooperativeActions/(double)totalActions;
  }
  else
  {
    cooperationLevel=0;
  }

  return (int)(cooperationLevel*100);
}

-increaseAlleleFrequency: (int) alleleNum
{
  //increase the frequency of an allele in the gene pool
  int t=[[alleleFrequencies atOffset: alleleNum] getIntData];
  [[alleleFrequencies atOffset: alleleNum] setIntData: t+1];

  return self;
}

-decreaseAlleleFrequency: (int) alleleNum
{
  //decrease the frequency of an allele in the gene pool
  int t=[[alleleFrequencies atOffset: alleleNum] getIntData];
  [[alleleFrequencies atOffset: alleleNum] setIntData: t-1];

  return self;
}

-getAlleleFrequency: (int) alleleNum
{
  //return the frequency of an allele in the gene pool
  return [alleleFrequencies atOffset: alleleNum];
}

-updateAverageAlleleFrequency: (int) alleleNum Increment: (int) alleleInc Population:(int) pop
{
  //increment the historical total of an allele in the gene pool
  int t;
  t=((double)alleleInc/pop)*100;
  [[alleleTotals atOffset: alleleNum] setIntData: [[alleleTotals atOffset: alleleNum] getIntData]+t];
  
  return self;
}

-(int)getAverageAlleleFrequency: (int) alleleNum
{
  //return the average historical frequency of an allele in the gene pool
  return [[alleleTotals atOffset: alleleNum] getIntData]/worldTime;
}

-increaseStrategyFrequency: (int) strategyNum;
{
  //increase the frequency of a strategy in the population

  int t=[[strategyFrequencies atOffset: strategyNum] getIntData];
  [[strategyFrequencies atOffset: strategyNum] setIntData: t+1];

  return self;
}

-decreaseStrategyFrequency: (int) strategyNum;
{
  //decrease the frequency of a strategy in the population

  int t=[[strategyFrequencies atOffset: strategyNum] getIntData];
  [[strategyFrequencies atOffset: strategyNum] setIntData: t-1];

  return self;
}

-getStrategyFrequency: (int) strategyNum;
{
  //return the frequency of a strategy in the population
  return [strategyFrequencies atOffset: strategyNum];
}

-updateCoopLevelMyAction: (int) action1 TheirAction: (int) action2
{
  //update the variables used in calculating the global cooperation level
  totalActions=totalActions+2;
  cooperativeActions=cooperativeActions+action1+action2;

  return self;
}

-(int)getStartEnergy
{
  //return the initial energy level allocated to new agents
  return startEnergy;
}

-(int)getLivingCost
{
  //return the energy cost of living for each time step
  return livingCost;
}

-(int)getWorldTime
{
  //return the world's age
  return worldTime;
}

-incrementWorldTime
{
  //increment the world's age
  worldTime++;

  return self;
}

-storeValues
{
  //store the global cooperation and relatedness levels for this time step
  id coopValue, relValue;
  int l;

  #ifdef consider_interaction_subset
    if (totalActions>0)
    {
  #endif
      coopValue=[IntegerData create: [self getZone]];
      [coopValue setIntData: (int)[self getCoopLevel]];
      [[cooperationValues getCollection] addLast: coopValue];
      relValue=[IntegerData create: [self getZone]];
      [relValue setIntData: (int)[self getGlobalRelatedness]];
      [[relatednessValues getCollection] addLast: relValue];
  #ifdef consider_interaction_subset
    }  
  #endif
  for (l=0; l<strategyLength; l++)
  {
    [self updateAverageAlleleFrequency: l Increment: [[self getAlleleFrequency: l] getIntData] Population: [[allAgents getCollection] getCount]];
  }

  return self;
}

-(int)getAverageCooperationLevel
{
  //return the average cooperation level over the world's history
  int total=0;
  id value;

  [cooperationValues setLoc: Start];
  while ((value=[cooperationValues next]))
  {
    total+=[value getIntData];
  }

  return (int)((double)total/(double)[[cooperationValues getCollection] getCount]);
}

-(double)getAverageNumberOfPartners
{
  // return average number of interaction partners per agent over the world's history
  return ((double)totalPartners/(double)totalInteractants);
}

-(int)getAverageRelatedness
{
  //return the average global relatedness over the world's history
  int total=0;
  id value;

  [relatednessValues setLoc: Start];
  while ((value=[relatednessValues next]))
  {
    total+=[value getIntData];
  }

  return (int)((double)total/(double)[[relatednessValues getCollection] getCount]);
}

-(double)getAverageInteractionLength
{
  //return the average length of iterated PD interactions in the world
  #ifdef pure_kin_selection
    return 0; //average interaction length not calculated for pure kin selection model as not of interest
  #else
    return (double)totalInteractionLength/(double)totalInteractions;
  #endif
}

-updateAverageLocalPopSize
{
  //update the average cell population size in the world
  int x, y;

  totalPopulationSize+=[[allAgents getCollection] getCount];
  for (x=0; x<xsize; x++)
  {
    for (y=0; y<ysize; y++)
    {
      if ([[[self getObjectAtX: x Y: y] getCollection] getCount]>0)
      {
        totalPopulatedCells++;
      }
    }
  }

  return self;
}

-(double)getAverageLocalPopSize
{
  //return the average cell population size in the world
  return (double)totalPopulationSize/(double)totalPopulatedCells;
}

-getWorldDisplay
{
  //return the array used to draw the world display
  return worldDisplay;
}

-resetDisplay
{
  //reset the world display
  int x,y;
  for (x=0; x<xsize; x++)
  {
    for (y=0; y<ysize; y++)
    {
      [worldDisplay putValue: 0 atX: x Y: y];
    }
  }

  return self;
}

-incrementTotalActionsAtX: (int) x Y: (int) y NumActions: (int) numActions
{
  [cellTotalActions putValue: [cellTotalActions getValueAtX: x Y: y]+numActions atX: x Y: y];

  return self;
}

-incrementTotalCooperativeActionsAtX: (int) x Y: (int) y NumCoopActions: (int) numCoopActions
{
  [cellTotalCooperativeActions putValue: [cellTotalCooperativeActions getValueAtX: x Y: y]+numCoopActions atX: x Y: y];

  return self;
}

-incrementTotalDiversityAtX: (int) x Y: (int) y Diversity: (double) diversity
{
  [cellTotalDiversity putValue: [cellTotalDiversity getValueAtX: x Y: y]+(diversity*100) atX: x Y: y];
  [cellDiversityDivisor putValue: [cellDiversityDivisor getValueAtX: x Y: y]+1 atX: x Y: y];

  return self;
}

-(double)getCooperationLevelAtX: (int) x Y: (int) y
{
  if ([cellTotalActions getValueAtX: x Y: y]==0)
  {
    return 0;
  }
  else
  {
    return (double)[cellTotalCooperativeActions getValueAtX: x Y: y]/(double)[cellTotalActions getValueAtX: x Y: y];
  }
}

-(double)getDiversityLevelAtX: (int) x Y: (int) y
{
  if ([cellDiversityDivisor getValueAtX: x Y: y]==0)
  {
    return 0;
  }
  else
  {
    return ((double)[cellTotalDiversity getValueAtX: x Y: y]/100)/(double)[cellDiversityDivisor getValueAtX: x Y: y];
  }
}

-resetCellTotalActions
{
  int x,y;
  for (x=0; x<xsize; x++)
  {
    for (y=0; y<ysize; y++)
    {
      [cellTotalActions putValue: 0 atX: x Y: y];
      [cellTotalCooperativeActions putValue: 0 atX: x Y: y];
    }
  }

  return self;
}

-resetCellTotalDiversity
{
  int x,y;
  for (x=0; x<xsize; x++)
  {
    for (y=0; y<ysize; y++)
    {
      [cellTotalDiversity putValue: 0 atX: x Y: y];
      [cellDiversityDivisor putValue: 0 atX: x Y: y];
    }
  }

  return self;
}

@end
