package XCSphere;

import java.lang.Math;

/**
 * This class implements the Multiplexer problem.
 * It provides randomly generated strings (the problem instances), determines if a classification
 * was correct, and which payoff is provided. The problem is implemented in a way that any problem length can be chosen. 
 * The class simply constructs the maximal multiplexer problem that fits in the specified problem length. The remaining bits are
 * (as the other bits) generated randomly in each problem instance but are irrelevant for the problem itself.
 *
 * Note that this could be greatly simplified by extending BooleanEnvironment,
 * although you would lose the layered payoff landscape.
 *
 * @author    Martin V. Butz
 * @version   XCSJava 1.0
 * @since     JDK1.1
 */
public class MPEnvironment extends BooleanEnvironment implements Environment
{
    /**
     * Defines if either a payoff landscape or a 1000/0 payoff is provided after the execution of a classification. 
     * The payoff Landscape is constructed as in Wilson's 
     * Classifier fitness based on accuracy paper (Evolutionary Computation Journal, 1995). 
     */
    public static boolean payoffLandscape;
    
    /*#####---- The following parameters are set internally dependent on the value of conLength and payoffLandscape ----#####*/

    /** 
     * Specifies the number of position bits in the multiplexer problem.
     */
    private int posBits;
    
    /**
     * Constructs the Multiplexer environment according to the specified problem length and chosen payoff type. 
     * Essentially the relevant constants for the environment are calculated here. 
     * Moreover, the problem array is generated
     *
     * @param length Specifies the problem length. The biggest Multiplexer problem is chosen that fits this length.
     * @param payoffMap Specifies if a payoff map should be provided or a 1000/0 payoff.
     * @param trackInstancesSeen Enables counting of unique problem instances seen (only suitable for small problems)
     * 
     */
    public MPEnvironment(int length, int payoffMap, boolean trackInstancesSeen)
    {
    super(length, trackInstancesSeen);
	conLength=length;
	double i;
	for(i=1.; i+Math.pow(2.,i)<=conLength; i++);//calculates the position bits in this problem
	posBits=(int)(i-1);

	if(posBits+Math.pow(2.,(double)posBits) != conLength)
	    System.out.println("There are additonally "+((int)(conLength-(posBits+Math.pow(2.,(double)posBits))))+" irrelevant Bits!");
	
	currentState = new char[conLength];
	
	if(payoffMap==0)
	    payoffLandscape=false;
	else
	    payoffLandscape=true;
	if(payoffLandscape){
	    maxPayoff=(int)(200 + 200 * Math.pow(2.,posBits));
	}else{
	    maxPayoff=1000;
	}

	correct=false;
	reset=false;
    }

    /**
     * Executes the action and determines the reward. (TODO: currently overrides BooleanEnvironment's implementation to use a payoff landcscape)
     * Distinguishes between the payoff landscape and the 0/1000 reward.
     *
     * @param action Specifies the classification.
     */
    public double executeAction(int action)
    {
	int place=posBits;
	for(int i=0; i<posBits; i++){
	    if(currentState[i]=='1'){
		place+=Math.pow(2., (double)(posBits-1-i));
	    }
	}
	int ret=0;
	if(action == Character.digit(currentState[place],10)){
	    correct=true;
	    if(payoffLandscape){
		ret = 300 + (place-posBits)*200 ;
		if(currentState[place]=='1')
		    ret+=100;
	    }else{
		ret = maxPayoff;
	    }
	}else{
	    correct=false;
	    if(payoffLandscape){
		ret = (place-posBits)*200;
		if(currentState[place] == '1')
		    ret+=100;
	    }else{
		ret = 0;
	    }
	}
	reset = true;
	return (double)ret;
    }
    
    public boolean isCorrect(int action)
    {
    	int place=posBits;
    	for(int i=0; i<posBits; i++){
    	    if(currentState[i]=='1'){
    		place+=Math.pow(2., (double)(posBits-1-i));
    	    }
    	}
    	if(action == Character.digit(currentState[place],10)){
    	    correct=true;
    	}else{
    	    correct=false;
    	}
    	return correct;
    }
}
