package XCSphere;

import uchicago.src.sim.util.Random;

/**
 * Provides additional constants and Gaussian distribution for hypersphere classifiers
 * May also be used to override provided pseudo-random number generator 
 *  
 * @author marshall
 *
 */
public class XCSphereConstants extends XCSConstants
{
	/** Split of hyperspheres vs hyperplanes (0 = all hyperspheres, 1 = all hyperplanes) */
	public static final double spherePlaneSplit = 0.5; 
	/** Mean radius for new hypersphere classifiers */
	public static final int newRadiusMean = 1;
	/** Variance for new hypersphere classifiers */
	public static final int newRadiusStdDev = 1;
	/** Probability of mutating hypersphere radius during classifier mutation */
	public static final double radiusMutationProb = 0.5;
	/** Variance for mutation of hypersphere classifiers */
	public static final int radiusMutationStdDev = 2;
	
	/**
     * Sets a random seed in order to randomize the pseudo random generators (replaces XCSConstants.drand()) 
     */
    public static void setSeed(long s)
    {
    	XCSConstants.setSeed(s);
    	initialiseRandomGenerators(s);
    }
	
	/**
	 * Initialises the random number generator (override to use your own random number generator)
	 *
	 */
	private static void initialiseRandomGenerators(long seed)
	{
		Random.setSeed(seed);
		Random.createUniform();
		Random.createNormal(0, 1);
	}
	
	/**
	 * Gets an integer from the normal distribution with specified mean and standard deviation (override to use your own random number generator)
	 * 
	 * @param mean
	 * @param stddev
	 * 
	 * @return integer from distribution
	 * 
	 */
	public static int nextIntFromNormalDist(int mean, int stddev)
	{
		return (int) Math.round(Random.normal.nextDouble(mean, stddev));
	}
	
	/**
	 * Gets a double from the uniform distribution (lowerBound, upperBound) (override to use your own random number generator)
	 * 
	 * @param lowerBound
	 * @param upperBound (> upperBound)
	 * 
	 * @return double from distribution
	 * 
	 */
	public static double nextDoubleFromUniformDist(double lowerBound, double upperBound)
	{
		if (upperBound <= lowerBound)
		{
			throw new IllegalArgumentException("nextDoubleFromUniformDist called with upperBound <= lowerBound");
		}
		
		return Random.uniform.nextDoubleFromTo(lowerBound, upperBound);
	}
	
    /**
     * Returns a random number in between zero and one (replaces XCSConstants.drand())
     */
    public static double drand()
    {
    	return XCSphereConstants.nextDoubleFromUniformDist(0, 1);
    }
}
